import { useState } from "react";
import { toast } from "sonner";
import { RemoteCommandRunner, CommandTarget } from "@/lib/devops/remoteCommandRunner";
import { LaravelRuntimeConfig, getArtisanCommands } from "@/lib/devops/laravelRuntimeConfig";

export interface ArtisanResult {
  command: string;
  success: boolean;
  output: string;
  error?: string;
  duration: number;
}

export const useLaravelPostDeploy = () => {
  const [running, setRunning] = useState(false);
  const [logs, setLogs] = useState<string[]>([]);
  const [results, setResults] = useState<ArtisanResult[]>([]);

  const addLog = (message: string) => {
    setLogs(prev => [...prev, `[${new Date().toLocaleTimeString()}] ${message}`]);
  };

  const runLaravelPostDeploy = async (
    target: CommandTarget,
    config: LaravelRuntimeConfig
  ): Promise<boolean> => {
    setRunning(true);
    setLogs([]);
    setResults([]);
    
    addLog("Starting Laravel post-deployment setup...");
    toast.info("Running Laravel artisan commands...");

    try {
      const runner = new RemoteCommandRunner(target);
      const commands = getArtisanCommands(config);
      const commandResults: ArtisanResult[] = [];

      let allSuccessful = true;

      for (const command of commands) {
        addLog(`Running: ${command}`);
        
        const result = await runner.runArtisanCommand(command);
        
        const artisanResult: ArtisanResult = {
          command,
          success: result.success,
          output: result.stdout,
          error: result.stderr,
          duration: result.duration
        };

        commandResults.push(artisanResult);

        if (result.success) {
          addLog(`✓ ${command} completed (${(result.duration / 1000).toFixed(2)}s)`);
          if (result.stdout) {
            result.stdout.split('\n').slice(0, 3).forEach(line => {
              if (line.trim()) addLog(`  ${line}`);
            });
          }
        } else {
          addLog(`✗ ${command} failed`);
          addLog(`  Error: ${result.stderr}`);
          allSuccessful = false;
          
          // Don't stop on error, continue with other commands
          // but mark deployment as needing attention
        }

        // Small delay between commands
        await new Promise(resolve => setTimeout(resolve, 500));
      }

      setResults(commandResults);

      if (allSuccessful) {
        addLog("✓ All Laravel post-deployment steps completed successfully");
        toast.success("Laravel setup completed");
        return true;
      } else {
        addLog("⚠ Some Laravel commands failed - deployment needs attention");
        toast.warning("Laravel setup completed with warnings");
        return false;
      }
    } catch (error: any) {
      addLog(`✗ Laravel post-deploy error: ${error.message}`);
      toast.error(`Laravel setup failed: ${error.message}`);
      return false;
    } finally {
      setRunning(false);
    }
  };

  const runSingleArtisanCommand = async (
    target: CommandTarget,
    command: string
  ): Promise<boolean> => {
    setRunning(true);
    addLog(`Running single command: ${command}`);

    try {
      const runner = new RemoteCommandRunner(target);
      const result = await runner.runArtisanCommand(command);

      const artisanResult: ArtisanResult = {
        command,
        success: result.success,
        output: result.stdout,
        error: result.stderr,
        duration: result.duration
      };

      setResults([artisanResult]);

      if (result.success) {
        addLog(`✓ Command completed`);
        toast.success("Artisan command executed");
        return true;
      } else {
        addLog(`✗ Command failed: ${result.stderr}`);
        toast.error("Artisan command failed");
        return false;
      }
    } catch (error: any) {
      addLog(`✗ Error: ${error.message}`);
      toast.error(`Error: ${error.message}`);
      return false;
    } finally {
      setRunning(false);
    }
  };

  const clearLogs = () => {
    setLogs([]);
    setResults([]);
  };

  return {
    runLaravelPostDeploy,
    runSingleArtisanCommand,
    running,
    logs,
    results,
    clearLogs
  };
};
