import { useState, useCallback } from "react";
import { useToast } from "@/hooks/use-toast";

export interface LogAnalysis {
  severity: "critical" | "error" | "warning" | "info";
  rootCause: string;
  suggestedFix: string;
  affectedFiles: string[];
  stackTrace?: string;
}

export const useLogAnalyzer = () => {
  const [analyzing, setAnalyzing] = useState(false);
  const [analysis, setAnalysis] = useState<LogAnalysis | null>(null);
  const { toast } = useToast();

  const analyzeLogs = useCallback(async (logs: string[]) => {
    setAnalyzing(true);
    try {
      // Simulate log analysis
      await new Promise(resolve => setTimeout(resolve, 1000));

      const result: LogAnalysis = {
        severity: "error",
        rootCause: "Component rendering issue detected",
        suggestedFix: "Check component lifecycle and props validation",
        affectedFiles: ["src/pages/Chat.tsx", "src/components/chat/MessageBubble.tsx"],
        stackTrace: logs.join("\n")
      };

      setAnalysis(result);
      
      toast({
        title: "Analysis Complete",
        description: "Root cause identified",
      });

      return result;
    } catch (error: any) {
      toast({
        title: "Analysis Failed",
        description: error.message,
        variant: "destructive",
      });
      return null;
    } finally {
      setAnalyzing(false);
    }
  }, [toast]);

  const clearAnalysis = useCallback(() => {
    setAnalysis(null);
  }, []);

  return {
    analyzeLogs,
    clearAnalysis,
    analysis,
    analyzing,
  };
};
