import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";

export interface Memory {
  id: string;
  user_id: string;
  memory_type: string;
  memory_key: string;
  memory_value: any;
  project_id?: string;
  created_at: string;
  updated_at: string;
  is_synced?: boolean;
  sync_priority?: number;
  local_only?: boolean;
}

export const useMemory = (projectId?: string) => {
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data: memories = [] } = useQuery({
    queryKey: ["user_memory", projectId],
    queryFn: async () => {
      let query = supabase.from("user_memory").select("*").order("updated_at", { ascending: false });

      if (projectId) {
        query = query.eq("project_id", projectId);
      }

      const { data, error } = await query;
      if (error) throw error;
      return data as Memory[];
    },
  });

  const saveMemory = useMutation({
    mutationFn: async ({ type, key, value, localOnly = false }: { type: string; key: string; value: any; localOnly?: boolean }) => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      // If offline or localOnly, store in IndexedDB first
      if (localOnly || !navigator.onLine) {
        // Store locally with sync flag
        const localMemory = {
          user_id: user.id,
          memory_type: type,
          memory_key: key,
          memory_value: value,
          project_id: projectId,
          is_synced: false,
          local_only: localOnly,
          updated_at: new Date().toISOString()
        };
        
        // Store in localStorage as fallback (should use IndexedDB in production)
        const localKey = `memory_${user.id}_${type}_${key}`;
        localStorage.setItem(localKey, JSON.stringify(localMemory));
        
        if (navigator.onLine && !localOnly) {
          // Sync to cloud
          const { data, error } = await supabase
            .from("user_memory")
            .upsert({
              user_id: user.id,
              memory_type: type,
              memory_key: key,
              memory_value: value,
              project_id: projectId,
              is_synced: true,
            })
            .select()
            .single();

          if (error) throw error;
          
          // Update local with synced flag
          localStorage.setItem(localKey, JSON.stringify({ ...localMemory, is_synced: true }));
          return data;
        }
        
        return localMemory;
      }

      const { data, error } = await supabase
        .from("user_memory")
        .upsert({
          user_id: user.id,
          memory_type: type,
          memory_key: key,
          memory_value: value,
          project_id: projectId,
          is_synced: true,
        })
        .select()
        .single();

      if (error) throw error;
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["user_memory", projectId] });
      toast({ title: "Memory saved" });
    },
    onError: (error: Error) => {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    },
  });

  const getMemory = async (type: string, key: string) => {
    // Try local first
    const { data: { user } } = await supabase.auth.getUser();
    if (user) {
      const localKey = `memory_${user.id}_${type}_${key}`;
      const localData = localStorage.getItem(localKey);
      if (localData) {
        return JSON.parse(localData);
      }
    }
    
    // Fallback to cloud
    return memories.find((m) => m.memory_type === type && m.memory_key === key);
  };

  const syncMemories = async () => {
    const { data: { user } } = await supabase.auth.getUser();
    if (!user || !navigator.onLine) return;

    // Get all unsynced local memories
    const localKeys = Object.keys(localStorage).filter(k => k.startsWith(`memory_${user.id}`));
    
    for (const localKey of localKeys) {
      try {
        const localData = JSON.parse(localStorage.getItem(localKey) || '{}');
        if (!localData.is_synced && !localData.local_only) {
          await supabase.from("user_memory").upsert({
            user_id: localData.user_id,
            memory_type: localData.memory_type,
            memory_key: localData.memory_key,
            memory_value: localData.memory_value,
            project_id: localData.project_id,
            is_synced: true,
          });
          
          // Update local
          localStorage.setItem(localKey, JSON.stringify({ ...localData, is_synced: true }));
        }
      } catch (error) {
        console.error("Sync error:", error);
      }
    }
  };

  return {
    memories,
    saveMemory,
    getMemory,
    syncMemories,
  };
};
