// Phase 3: Module Builder Hook

import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { supabase } from '@/integrations/supabase/client';
import { useToast } from '@/hooks/use-toast';
import { ModuleBlueprint } from '@/types/moduleBlueprint';
import { builderRunner } from '@/core/moduleBuilder/BuilderRunner';

export const useModuleBuilder = () => {
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch all blueprints
  const { data: blueprints = [], isLoading } = useQuery({
    queryKey: ['sf_module_blueprints'],
    queryFn: async () => {
      const { data, error } = await supabase
        .from('sf_module_blueprints')
        .select('*')
        .order('created_at', { ascending: false });

      if (error) throw error;
      return data as any[];
    },
  });

  // Create blueprint from natural language
  const createFromDescription = useMutation({
    mutationFn: async ({ description, locale = 'en' }: { description: string; locale?: string }) => {
      // Generate the module
      const result = await builderRunner.buildModule(description, locale);

      if (!result.success) {
        throw new Error(result.error || 'Failed to generate module');
      }

      // Save to database
      const { data: user } = await supabase.auth.getUser();
      if (!user.user) throw new Error('Not authenticated');

      const { data, error } = await supabase
        .from('sf_module_blueprints')
        .insert({
          created_by: user.user.id,
          name: result.blueprint.name,
          slug: result.blueprint.slug,
          description: result.blueprint.description,
          natural_description: description,
          entities: result.blueprint.entities as any,
          relationships: result.blueprint.relationships as any,
          ui_pages: result.blueprint.uiPages as any,
          api_routes: result.blueprint.apiRoutes as any,
          backend_stack: result.blueprint.backendStack,
          frontend_stack: result.blueprint.frontendStack,
          permissions: result.blueprint.permissions as any,
          menu_items: result.blueprint.menuItems as any,
          complexity: result.blueprint.complexity,
          status: result.blueprint.status,
          generated_files: result.files as any,
          version: result.blueprint.version,
          tags: result.blueprint.tags || [],
        } as any)
        .select()
        .single();

      if (error) throw error;
      return { ...result, id: data.id };
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['sf_module_blueprints'] });
      toast({ title: 'Module blueprint created successfully' });
    },
    onError: (error: Error) => {
      toast({ 
        title: 'Failed to create module', 
        description: error.message, 
        variant: 'destructive' 
      });
    },
  });

  // Update blueprint
  const updateBlueprint = useMutation({
    mutationFn: async ({ id, updates }: { id: string; updates: Partial<ModuleBlueprint> }) => {
      const { data, error } = await supabase
        .from('sf_module_blueprints')
        .update(updates as any)
        .eq('id', id)
        .select()
        .single();

      if (error) throw error;
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['sf_module_blueprints'] });
      toast({ title: 'Blueprint updated successfully' });
    },
    onError: (error: Error) => {
      toast({ title: 'Error', description: error.message, variant: 'destructive' });
    },
  });

  // Apply blueprint (activate module)
  const applyBlueprint = useMutation({
    mutationFn: async (blueprintId: string) => {
      // Get blueprint
      const { data: blueprint, error: fetchError } = await supabase
        .from('sf_module_blueprints')
        .select('*')
        .eq('id', blueprintId)
        .single();

      if (fetchError) throw fetchError;

      // Create module entry
      const { data: module, error: moduleError } = await supabase
        .from('sf_modules')
        .insert({
          slug: blueprint.slug,
          name: blueprint.name,
          description: blueprint.description,
          route: `/${blueprint.slug}`,
          frontend_entry: `src/pages/${blueprint.name}List.tsx`,
          is_active: true,
          blueprint_id: blueprintId,
          category: 'Generated',
          version: blueprint.version,
        })
        .select()
        .single();

      if (moduleError) throw moduleError;

      // Update blueprint status
      const { error: updateError } = await supabase
        .from('sf_module_blueprints')
        .update({ status: 'applied' })
        .eq('id', blueprintId);

      if (updateError) throw updateError;

      return module;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['sf_module_blueprints'] });
      queryClient.invalidateQueries({ queryKey: ['sf_modules'] });
      toast({ 
        title: 'Module activated successfully', 
        description: 'The module is now live and accessible' 
      });
    },
    onError: (error: Error) => {
      toast({ title: 'Error', description: error.message, variant: 'destructive' });
    },
  });

  // Delete blueprint
  const deleteBlueprint = useMutation({
    mutationFn: async (id: string) => {
      const { error } = await supabase
        .from('sf_module_blueprints')
        .delete()
        .eq('id', id);

      if (error) throw error;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['sf_module_blueprints'] });
      toast({ title: 'Blueprint deleted successfully' });
    },
    onError: (error: Error) => {
      toast({ title: 'Error', description: error.message, variant: 'destructive' });
    },
  });

  return {
    blueprints,
    isLoading,
    createFromDescription,
    updateBlueprint,
    applyBlueprint,
    deleteBlueprint,
  };
};
