import { useState, useCallback } from "react";
import { useToast } from "@/hooks/use-toast";

export interface ModuleConfig {
  name: string;
  slug: string;
  description: string;
  category: string;
  isPublic: boolean;
  hasAPI: boolean;
  hasAuth: boolean;
  hasCRUD: boolean;
  features: string[];
}

export interface GeneratedModule {
  id: string;
  config: ModuleConfig;
  files: {
    frontend: string[];
    backend: string[];
    database: string[];
  };
  status: "pending" | "generating" | "completed" | "failed";
  created: string;
}

export const useModuleBuilderPro = () => {
  const [generating, setGenerating] = useState(false);
  const [modules, setModules] = useState<GeneratedModule[]>([]);
  const { toast } = useToast();

  const moduleTemplates: Record<string, Partial<ModuleConfig>> = {
    academy: {
      category: "education",
      features: ["classes", "subjects", "exams", "students", "teachers"],
      hasAPI: true,
      hasAuth: true,
      hasCRUD: true,
    },
    doctor: {
      category: "healthcare",
      features: ["appointments", "prescriptions", "patients", "doctors"],
      hasAPI: true,
      hasAuth: true,
      hasCRUD: true,
    },
    news: {
      category: "media",
      features: ["articles", "categories", "comments", "authors"],
      hasAPI: true,
      hasAuth: false,
      hasCRUD: true,
    },
    bazar: {
      category: "ecommerce",
      features: ["products", "orders", "merchants", "payments"],
      hasAPI: true,
      hasAuth: true,
      hasCRUD: true,
    },
    pay: {
      category: "finance",
      features: ["transactions", "wallets", "transfers", "history"],
      hasAPI: true,
      hasAuth: true,
      hasCRUD: true,
    },
  };

  const generateModule = useCallback(async (config: ModuleConfig) => {
    setGenerating(true);
    try {
      const moduleId = `module-${Date.now()}`;
      
      const newModule: GeneratedModule = {
        id: moduleId,
        config,
        files: {
          frontend: [],
          backend: [],
          database: [],
        },
        status: "generating",
        created: new Date().toISOString(),
      };

      setModules(prev => [newModule, ...prev]);

      // Simulate module generation
      await new Promise(resolve => setTimeout(resolve, 2000));

      // Generate frontend files
      const frontendFiles = [
        `src/pages/${config.slug}/${config.name}.tsx`,
        `src/components/${config.slug}/${config.name}List.tsx`,
        `src/components/${config.slug}/${config.name}Card.tsx`,
        `src/hooks/use${config.name}.tsx`,
      ];

      // Generate backend files if hasAPI
      const backendFiles = config.hasAPI ? [
        `backend/app/Http/Controllers/${config.name}Controller.php`,
        `backend/app/Models/${config.name}.php`,
        `backend/app/Http/Requests/Store${config.name}Request.php`,
        `backend/app/Http/Requests/Update${config.name}Request.php`,
        `backend/routes/api.php`,
      ] : [];

      // Generate database files
      const databaseFiles = [
        `backend/database/migrations/create_${config.slug}_table.php`,
        `backend/database/seeders/${config.name}Seeder.php`,
      ];

      if (config.hasAuth) {
        backendFiles.push(`backend/app/Policies/${config.name}Policy.php`);
      }

      setModules(prev =>
        prev.map(m =>
          m.id === moduleId
            ? {
                ...m,
                files: {
                  frontend: frontendFiles,
                  backend: backendFiles,
                  database: databaseFiles,
                },
                status: "completed",
              }
            : m
        )
      );

      toast({
        title: "Module Generated",
        description: `${config.name} module created successfully`,
      });

      return newModule;
    } catch (error: any) {
      toast({
        title: "Generation Failed",
        description: error.message,
        variant: "destructive",
      });
      return null;
    } finally {
      setGenerating(false);
    }
  }, [toast]);

  const getTemplate = useCallback((templateName: string): Partial<ModuleConfig> | null => {
    return moduleTemplates[templateName] || null;
  }, []);

  const getSuggestedModules = useCallback(() => {
    return [
      { name: "Smart Academy", slug: "smart-academy", template: "academy" },
      { name: "Smart Doctor", slug: "smart-doctor", template: "doctor" },
      { name: "Smart News", slug: "smart-news", template: "news" },
      { name: "Smart Bazar", slug: "smart-bazar", template: "bazar" },
      { name: "Smart Pay", slug: "smart-pay", template: "pay" },
    ];
  }, []);

  const repairModule = useCallback(async (moduleId: string) => {
    try {
      await new Promise(resolve => setTimeout(resolve, 1000));
      
      setModules(prev =>
        prev.map(m =>
          m.id === moduleId && m.status === "failed"
            ? { ...m, status: "completed" }
            : m
        )
      );

      toast({
        title: "Module Repaired",
        description: "Module has been fixed successfully",
      });

      return true;
    } catch (error: any) {
      toast({
        title: "Repair Failed",
        description: error.message,
        variant: "destructive",
      });
      return false;
    }
  }, [toast]);

  return {
    generateModule,
    getTemplate,
    getSuggestedModules,
    repairModule,
    modules,
    generating,
  };
};
