import { useQuery, useMutation } from "@tanstack/react-query";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";

export interface ModuleTemplate {
  id: string;
  name: string;
  slug: string;
  description: string;
  category: string;
  template_config: any;
  is_multi_tenant: boolean;
  created_at: string;
  updated_at: string;
}

export const useModuleGenerator = () => {
  const { toast } = useToast();

  const { data: templates = [], isLoading } = useQuery({
    queryKey: ["module_templates"],
    queryFn: async () => {
      const { data, error } = await supabase
        .from("module_templates")
        .select("*")
        .order("name");

      if (error) throw error;
      return data as ModuleTemplate[];
    },
  });

  const generateModule = useMutation({
    mutationFn: async ({ templateSlug, projectId, customName }: {
      templateSlug: string;
      projectId: string;
      customName?: string;
    }) => {
      const { data, error } = await supabase.functions.invoke("module-generator", {
        body: { templateSlug, projectId, customName },
      });

      if (error) throw error;
      return data;
    },
    onSuccess: () => {
      toast({ title: "Module generated successfully" });
    },
    onError: (error: Error) => {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    },
  });

  return {
    templates,
    isLoading,
    generateModule,
  };
};
