import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";

export interface Module {
  id: string;
  name: string;
  slug: string;
  description: string | null;
  category: string;
  module_config: {
    permissions: string[];
    actions: string[];
    ai_suggestions: string[];
  };
  created_at: string;
  updated_at: string;
}

export const useModules = () => {
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data: modules, isLoading } = useQuery({
    queryKey: ["code_modules"],
    queryFn: async () => {
      const { data, error } = await supabase
        .from("code_modules")
        .select("*")
        .order("created_at", { ascending: false });

      if (error) throw error;
      return (data || []).map(item => ({
        ...item,
        module_config: item.module_config as { permissions: string[]; actions: string[]; ai_suggestions: string[] }
      })) as Module[];
    },
  });

  const createModule = useMutation({
    mutationFn: async (moduleData: {
      name: string;
      slug: string;
      description: string;
      category: string;
    }) => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      const { data, error } = await supabase
        .from("code_modules")
        .insert({
          ...moduleData,
          created_by: user.id,
        })
        .select()
        .single();

      if (error) throw error;
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["code_modules"] });
      toast({
        title: "Module created",
        description: "Module files generated successfully",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Error",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const updateModule = useMutation({
    mutationFn: async ({
      id,
      ...updates
    }: Partial<Module> & { id: string }) => {
      const { data, error } = await supabase
        .from("code_modules")
        .update(updates)
        .eq("id", id)
        .select()
        .single();

      if (error) throw error;
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["code_modules"] });
      toast({
        title: "Module updated",
        description: "Module updated successfully",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Error",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const deleteModule = useMutation({
    mutationFn: async (id: string) => {
      const { error } = await supabase
        .from("code_modules")
        .delete()
        .eq("id", id);

      if (error) throw error;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["code_modules"] });
      toast({
        title: "Module deleted",
        description: "Module deleted successfully",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Error",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  return {
    modules,
    isLoading,
    createModule,
    updateModule,
    deleteModule,
  };
};
