import { useQuery } from "@tanstack/react-query";
import { supabase } from "@/integrations/supabase/client";
import { useLanguage } from "./useLanguage";

export interface OfflineBrainEntry {
  id: string;
  question: string;
  answer: string;
  category: string;
  language: string;
  updated_at: string;
}

export const useOfflineBrain = () => {
  const { language } = useLanguage();

  const { data: offlineBrain = [] } = useQuery({
    queryKey: ["offline_brain", language],
    queryFn: async () => {
      const { data, error } = await supabase
        .from("offline_brain")
        .select("*")
        .eq("language", language)
        .order("updated_at", { ascending: false });
      
      if (error) throw error;
      return data as OfflineBrainEntry[];
    },
  });

  const searchOfflineResponse = async (question: string): Promise<string | null> => {
    try {
      const { data, error } = await supabase.functions.invoke("offline-engine", {
        body: { question, language },
      });

      if (error) throw error;
      return data?.answer || null;
    } catch (error) {
      console.error("Error searching offline brain:", error);
      return null;
    }
  };

  const saveOfflineResponse = async (question: string, answer: string, category: string = "general") => {
    try {
      const { error } = await supabase.from("offline_brain").insert({
        question,
        answer,
        category,
        language,
      });

      if (error) throw error;
    } catch (error) {
      console.error("Error saving offline response:", error);
    }
  };

  return {
    offlineBrain,
    searchOfflineResponse,
    saveOfflineResponse,
  };
};
