import { useCallback } from "react";
import offlineTranslations from "@/lib/i18n/offlineTranslations.json";

export const useOfflineIdentity = () => {
  // Check if question matches identity patterns
  const checkIdentityMatch = useCallback((question: string, language: string): string | null => {
    const normalizedQuestion = question.toLowerCase().trim();
    const identity = offlineTranslations.identity as any; // Type assertion to handle dynamic properties

    // HIGHEST PRIORITY: Check name question
    const namePatterns = identity.name_question?.[language] || identity.name_question?.en || [];
    if (namePatterns.some((pattern: string) => normalizedQuestion.includes(pattern.toLowerCase()))) {
      return identity.name_answer?.[language] || identity.name_answer?.en || "My name is AVA — I am your Smart Friend.";
    }

    // Check creator question
    const creatorPatterns = identity.creator_question[language as keyof typeof identity.creator_question] || [];
    if (creatorPatterns.some((pattern: string) => normalizedQuestion.includes(pattern.toLowerCase()))) {
      return identity.creator_answer[language as keyof typeof identity.creator_answer] || identity.creator_answer.en;
    }

    // Check know Tasnim/Nazrul
    const knowPatterns = identity.know_tasnim[language as keyof typeof identity.know_tasnim] || [];
    if (knowPatterns.some((pattern: string) => normalizedQuestion.includes(pattern.toLowerCase()))) {
      return identity.know_tasnim_answer[language as keyof typeof identity.know_tasnim_answer] || identity.know_tasnim_answer.en;
    }

    // Check location question
    const locationPatterns = identity.location_question[language as keyof typeof identity.location_question] || [];
    if (locationPatterns.some((pattern: string) => normalizedQuestion.includes(pattern.toLowerCase()))) {
      return identity.location_answer[language as keyof typeof identity.location_answer] || identity.location_answer.en;
    }

    return null;
  }, []);

  // Check for basic greetings
  const checkGreeting = useCallback((question: string, language: string): string | null => {
    const normalizedQuestion = question.toLowerCase().trim();
    
    // Check prohibited greetings first (religious greetings) - return polite neutral response
    const prohibitedPatterns = offlineTranslations.prohibited_greetings[language as keyof typeof offlineTranslations.prohibited_greetings] || [];
    if (prohibitedPatterns.some((pattern: string) => normalizedQuestion.includes(pattern.toLowerCase()))) {
      return offlineTranslations.religious_greeting_response[language as keyof typeof offlineTranslations.religious_greeting_response] || offlineTranslations.religious_greeting_response.en;
    }

    const greetingPatterns = offlineTranslations.basic_greeting[language as keyof typeof offlineTranslations.basic_greeting] || [];
    
    if (greetingPatterns.some((pattern: string) => normalizedQuestion === pattern.toLowerCase())) {
      return offlineTranslations.greeting_response[language as keyof typeof offlineTranslations.greeting_response] || offlineTranslations.greeting_response.en;
    }

    return null;
  }, []);

  // Check for help requests
  const checkHelp = useCallback((question: string, language: string): string | null => {
    const normalizedQuestion = question.toLowerCase().trim();
    const helpPatterns = offlineTranslations.help_question[language as keyof typeof offlineTranslations.help_question] || [];
    
    if (helpPatterns.some((pattern: string) => normalizedQuestion.includes(pattern.toLowerCase()))) {
      return offlineTranslations.help[language as keyof typeof offlineTranslations.help] || offlineTranslations.help.en;
    }

    return null;
  }, []);

  // Main identity check (runs before any API call)
  const getOfflineResponse = useCallback((question: string, language: string = "bn"): string | null => {
    // Priority order: Name > Identity > Greeting > Help (name check is now inside checkIdentityMatch)
    const identityAnswer = checkIdentityMatch(question, language);
    if (identityAnswer) return identityAnswer;

    const greetingAnswer = checkGreeting(question, language);
    if (greetingAnswer) return greetingAnswer;

    const helpAnswer = checkHelp(question, language);
    if (helpAnswer) return helpAnswer;

    return null;
  }, [checkIdentityMatch, checkGreeting, checkHelp]);

  return {
    getOfflineResponse,
    checkIdentityMatch,
  };
};
