import { useState, useEffect, useCallback } from "react";
import { toast } from "sonner";

interface CachedResponse {
  question: string;
  answer: string;
  timestamp: number;
  language: string;
}

interface QueuedMessage {
  id: string;
  content: string;
  timestamp: number;
  status: "queued" | "sending" | "sent" | "failed";
}

export const useOfflineMode = () => {
  const [isOnline, setIsOnline] = useState(navigator.onLine);
  const [queuedMessages, setQueuedMessages] = useState<QueuedMessage[]>([]);
  const CACHE_KEY = "smart_friend_offline_cache";
  const QUEUE_KEY = "smart_friend_message_queue";
  const CACHE_EXPIRY = 7 * 24 * 60 * 60 * 1000; // 7 days

  // Network status monitoring
  useEffect(() => {
    const handleOnline = () => {
      setIsOnline(true);
      toast.success("Back online! Syncing messages...");
      processQueue();
    };

    const handleOffline = () => {
      setIsOnline(false);
      toast.warning("You're offline. Messages will be queued.");
    };

    window.addEventListener("online", handleOnline);
    window.addEventListener("offline", handleOffline);

    // Load queued messages from localStorage
    loadQueue();

    return () => {
      window.removeEventListener("online", handleOnline);
      window.removeEventListener("offline", handleOffline);
    };
  }, []);

  // Load queue from localStorage
  const loadQueue = () => {
    try {
      const stored = localStorage.getItem(QUEUE_KEY);
      if (stored) {
        setQueuedMessages(JSON.parse(stored));
      }
    } catch (error) {
      console.error("Error loading queue:", error);
    }
  };

  // Save queue to localStorage
  const saveQueue = (queue: QueuedMessage[]) => {
    try {
      localStorage.setItem(QUEUE_KEY, JSON.stringify(queue));
    } catch (error) {
      console.error("Error saving queue:", error);
    }
  };

  // Cache a response
  const cacheResponse = useCallback((question: string, answer: string, language: string) => {
    try {
      const cache = getCachedResponses();
      const newEntry: CachedResponse = {
        question: question.toLowerCase().trim(),
        answer,
        timestamp: Date.now(),
        language,
      };
      
      // Keep only last 100 entries
      const updatedCache = [newEntry, ...cache].slice(0, 100);
      localStorage.setItem(CACHE_KEY, JSON.stringify(updatedCache));
    } catch (error) {
      console.error("Error caching response:", error);
    }
  }, []);

  // Get cached responses
  const getCachedResponses = (): CachedResponse[] => {
    try {
      const stored = localStorage.getItem(CACHE_KEY);
      if (!stored) return [];
      
      const cache: CachedResponse[] = JSON.parse(stored);
      const now = Date.now();
      
      // Filter out expired entries
      return cache.filter(entry => now - entry.timestamp < CACHE_EXPIRY);
    } catch (error) {
      console.error("Error getting cache:", error);
      return [];
    }
  };

  // Find cached answer
  const findCachedAnswer = useCallback((question: string, language: string): string | null => {
    const cache = getCachedResponses();
    const normalizedQuestion = question.toLowerCase().trim();
    
    // Exact match
    const exactMatch = cache.find(
      entry => entry.question === normalizedQuestion && entry.language === language
    );
    if (exactMatch) return exactMatch.answer;
    
    // Fuzzy match (contains key words)
    const fuzzyMatch = cache.find(entry => {
      if (entry.language !== language) return false;
      const words = normalizedQuestion.split(" ");
      return words.some(word => word.length > 3 && entry.question.includes(word));
    });
    
    return fuzzyMatch?.answer || null;
  }, []);

  // Queue a message
  const queueMessage = useCallback((content: string) => {
    const newMessage: QueuedMessage = {
      id: `msg_${Date.now()}_${Math.random()}`,
      content,
      timestamp: Date.now(),
      status: "queued",
    };
    
    const updatedQueue = [...queuedMessages, newMessage];
    setQueuedMessages(updatedQueue);
    saveQueue(updatedQueue);
    
    return newMessage.id;
  }, [queuedMessages]);

  // Process queued messages
  const processQueue = useCallback(async () => {
    if (!navigator.onLine || queuedMessages.length === 0) return;
    
    // Process messages one by one
    for (const message of queuedMessages) {
      if (message.status === "queued") {
        // Mark as sending
        updateMessageStatus(message.id, "sending");
        
        // The actual sending will be handled by the Chat component
        // This just marks them as ready to send
        await new Promise(resolve => setTimeout(resolve, 100));
      }
    }
  }, [queuedMessages]);

  // Update message status
  const updateMessageStatus = (id: string, status: QueuedMessage["status"]) => {
    setQueuedMessages(prev => {
      const updated = prev.map(msg => 
        msg.id === id ? { ...msg, status } : msg
      );
      saveQueue(updated);
      return updated;
    });
  };

  // Remove message from queue
  const removeFromQueue = (id: string) => {
    setQueuedMessages(prev => {
      const updated = prev.filter(msg => msg.id !== id);
      saveQueue(updated);
      return updated;
    });
  };

  // Clear all cache
  const clearCache = () => {
    localStorage.removeItem(CACHE_KEY);
    toast.success("Cache cleared");
  };

  // Get fallback response
  const getFallbackResponse = (language: string): string => {
    if (language === "bn") {
      return "দুঃখিত, আমি এই মুহূর্তে অফলাইনে আছি। আপনার বার্তাটি সংরক্ষিত হয়েছে এবং অনলাইনে ফিরে এলে উত্তর দেওয়া হবে।";
    }
    return "Sorry, I'm currently offline. Your message has been saved and will be answered when back online.";
  };

  return {
    isOnline,
    queuedMessages,
    cacheResponse,
    findCachedAnswer,
    queueMessage,
    processQueue,
    updateMessageStatus,
    removeFromQueue,
    clearCache,
    getFallbackResponse,
  };
};

