import { useState, useEffect } from 'react';

export interface OnboardingData {
  completed: boolean;
  role: 'owner' | 'ava' | 'guest';
  language: string;
  region: string;
  pinnedModules: string[];
  username?: string;
}

const DEFAULT_ONBOARDING: OnboardingData = {
  completed: false,
  role: 'owner',
  language: 'bn',
  region: 'BD',
  pinnedModules: [],
};

export const useOnboarding = () => {
  const [onboarding, setOnboarding] = useState<OnboardingData>(() => {
    const stored = localStorage.getItem('smart_friend_onboarding');
    if (stored) {
      try {
        return JSON.parse(stored);
      } catch {
        return DEFAULT_ONBOARDING;
      }
    }
    return DEFAULT_ONBOARDING;
  });

  const saveOnboarding = (data: Partial<OnboardingData>) => {
    const updated = { ...onboarding, ...data };
    setOnboarding(updated);
    localStorage.setItem('smart_friend_onboarding', JSON.stringify(updated));
  };

  const completeOnboarding = (data: Partial<OnboardingData>) => {
    saveOnboarding({ ...data, completed: true });
  };

  const resetOnboarding = () => {
    setOnboarding(DEFAULT_ONBOARDING);
    localStorage.removeItem('smart_friend_onboarding');
  };

  const isFirstLaunch = !onboarding.completed;

  return {
    onboarding,
    saveOnboarding,
    completeOnboarding,
    resetOnboarding,
    isFirstLaunch,
  };
};
