import { useState } from "react";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";

export type PaymentProvider = "bkash" | "nagad" | "stripe";

interface PaymentData {
  amount: number;
  provider: PaymentProvider;
  purpose: "wallet" | "subscription" | "payout";
  metadata?: Record<string, any>;
}

export const usePaymentGateway = () => {
  const [isProcessing, setIsProcessing] = useState(false);
  const { toast } = useToast();

  const initiatePayment = async (paymentData: PaymentData) => {
    setIsProcessing(true);
    
    try {
      const { data, error } = await supabase.functions.invoke("payment-gateway", {
        body: paymentData,
      });

      if (error) throw error;

      // Open payment URL in popup or redirect
      if (data.paymentUrl) {
        window.open(data.paymentUrl, "_blank", "width=600,height=700");
      }

      return data;
    } catch (error) {
      console.error("Payment error:", error);
      toast({
        title: "Payment failed",
        description: error instanceof Error ? error.message : "Unknown error",
        variant: "destructive",
      });
      throw error;
    } finally {
      setIsProcessing(false);
    }
  };

  const verifyPayment = async (transactionId: string) => {
    try {
      const { data, error } = await supabase.functions.invoke("verify-payment", {
        body: { transactionId },
      });

      if (error) throw error;

      if (data.status === "success") {
        toast({ title: "Payment verified successfully" });
      }

      return data;
    } catch (error) {
      console.error("Verification error:", error);
      toast({
        title: "Verification failed",
        description: error instanceof Error ? error.message : "Unknown error",
        variant: "destructive",
      });
      throw error;
    }
  };

  return {
    isProcessing,
    initiatePayment,
    verifyPayment,
  };
};
