import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";

export interface ProjectFile {
  id: string;
  user_id: string;
  project_id: string;
  file_path: string;
  file_name: string;
  file_type: "file" | "folder";
  content?: string;
  parent_id?: string;
  created_at: string;
  updated_at: string;
}

export const useProjectFiles = (projectId?: string) => {
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data: files = [], isLoading } = useQuery({
    queryKey: ["project_files", projectId],
    queryFn: async () => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      let query = supabase
        .from("project_files")
        .select("*")
        .eq("user_id", user.id)
        .order("file_path");

      if (projectId) {
        query = query.eq("project_id", projectId);
      }

      const { data, error } = await query;
      if (error) throw error;
      return data as ProjectFile[];
    },
    enabled: !!projectId,
  });

  const createFile = useMutation({
    mutationFn: async ({ fileName, filePath, fileType, content, parentId }: {
      fileName: string;
      filePath: string;
      fileType: "file" | "folder";
      content?: string;
      parentId?: string;
    }) => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      const { data, error } = await supabase
        .from("project_files")
        .insert({
          user_id: user.id,
          project_id: projectId || crypto.randomUUID(),
          file_name: fileName,
          file_path: filePath,
          file_type: fileType,
          content: content || "",
          parent_id: parentId,
        })
        .select()
        .single();

      if (error) throw error;
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["project_files", projectId] });
      toast({ title: "File created successfully" });
    },
    onError: (error: Error) => {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    },
  });

  const updateFile = useMutation({
    mutationFn: async ({ id, content, fileName }: { id: string; content?: string; fileName?: string }) => {
      const updates: any = {};
      if (content !== undefined) updates.content = content;
      if (fileName !== undefined) updates.file_name = fileName;

      const { data, error } = await supabase
        .from("project_files")
        .update(updates)
        .eq("id", id)
        .select()
        .single();

      if (error) throw error;
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["project_files", projectId] });
      toast({ title: "File updated successfully" });
    },
    onError: (error: Error) => {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    },
  });

  const deleteFile = useMutation({
    mutationFn: async (id: string) => {
      const { error } = await supabase
        .from("project_files")
        .delete()
        .eq("id", id);

      if (error) throw error;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["project_files", projectId] });
      toast({ title: "File deleted successfully" });
    },
    onError: (error: Error) => {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    },
  });

  return {
    files,
    isLoading,
    createFile,
    updateFile,
    deleteFile,
  };
};
