import { useState } from "react";
import { toast } from "sonner";

export interface ProjectStructure {
  name: string;
  framework: "nextjs" | "react-spa";
  files: Record<string, string>;
  structure: string[];
}

export const useProjectGenerator = () => {
  const [generating, setGenerating] = useState(false);
  const [project, setProject] = useState<ProjectStructure | null>(null);

  const generateNextJS = (projectName: string, components: string[]): ProjectStructure => {
    const files: Record<string, string> = {
      "package.json": JSON.stringify({
        name: projectName,
        version: "1.0.0",
        scripts: {
          dev: "next dev",
          build: "next build",
          start: "next start",
        },
        dependencies: {
          next: "latest",
          react: "^18.3.1",
          "react-dom": "^18.3.1",
        },
      }, null, 2),
      "next.config.js": `/** @type {import('next').NextConfig} */
const nextConfig = {
  reactStrictMode: true,
}

module.exports = nextConfig`,
      "pages/index.tsx": `import Head from 'next/head'

export default function Home() {
  return (
    <>
      <Head>
        <title>${projectName}</title>
        <meta name="description" content="Generated with Smart Friend" />
      </Head>
      <main>
        <h1>Welcome to ${projectName}</h1>
      </main>
    </>
  )
}`,
      "pages/_app.tsx": `import type { AppProps } from 'next/app'
import '../styles/globals.css'

export default function App({ Component, pageProps }: AppProps) {
  return <Component {...pageProps} />
}`,
      "styles/globals.css": `@tailwind base;
@tailwind components;
@tailwind utilities;

* {
  box-sizing: border-box;
  padding: 0;
  margin: 0;
}`,
      "README.md": `# ${projectName}

This project was generated by Smart Friend Website Cloner.

## Getting Started

\`\`\`bash
npm install
npm run dev
\`\`\`

Open [http://localhost:3000](http://localhost:3000)`,
    };

    components.forEach(comp => {
      files[`components/${comp}`] = `// ${comp} component`;
    });

    return {
      name: projectName,
      framework: "nextjs",
      files,
      structure: Object.keys(files),
    };
  };

  const generateReactSPA = (projectName: string, components: string[]): ProjectStructure => {
    const files: Record<string, string> = {
      "package.json": JSON.stringify({
        name: projectName,
        version: "1.0.0",
        scripts: {
          dev: "vite",
          build: "vite build",
          preview: "vite preview",
        },
        dependencies: {
          react: "^18.3.1",
          "react-dom": "^18.3.1",
        },
        devDependencies: {
          "@vitejs/plugin-react": "^4.2.0",
          vite: "^5.0.0",
        },
      }, null, 2),
      "index.html": `<!DOCTYPE html>
<html lang="en">
  <head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0" />
    <title>${projectName}</title>
  </head>
  <body>
    <div id="root"></div>
    <script type="module" src="/src/main.tsx"></script>
  </body>
</html>`,
      "src/main.tsx": `import React from 'react'
import ReactDOM from 'react-dom/client'
import App from './App'
import './index.css'

ReactDOM.createRoot(document.getElementById('root')!).render(
  <React.StrictMode>
    <App />
  </React.StrictMode>,
)`,
      "src/App.tsx": `function App() {
  return (
    <div className="app">
      <h1>Welcome to ${projectName}</h1>
    </div>
  )
}

export default App`,
      "src/index.css": `@tailwind base;
@tailwind components;
@tailwind utilities;`,
      "README.md": `# ${projectName}

This project was generated by Smart Friend Website Cloner.

## Getting Started

\`\`\`bash
npm install
npm run dev
\`\`\``,
    };

    components.forEach(comp => {
      files[`src/components/${comp}`] = `// ${comp} component`;
    });

    return {
      name: projectName,
      framework: "react-spa",
      files,
      structure: Object.keys(files),
    };
  };

  const generateProject = async (
    projectName: string,
    framework: "nextjs" | "react-spa",
    components: string[]
  ): Promise<ProjectStructure | null> => {
    setGenerating(true);
    toast.info("Generating project structure...");

    try {
      await new Promise(resolve => setTimeout(resolve, 2000));

      const project = framework === "nextjs"
        ? generateNextJS(projectName, components)
        : generateReactSPA(projectName, components);

      setProject(project);
      toast.success(`Project ${projectName} generated!`);
      return project;

    } catch (error: any) {
      toast.error(`Failed to generate project: ${error.message}`);
      return null;
    } finally {
      setGenerating(false);
    }
  };

  return {
    generateProject,
    generating,
    project,
  };
};
