import { useState, useCallback, useRef } from "react";
import { useToast } from "@/hooks/use-toast";

export interface RuntimeResult {
  output: string;
  error?: string;
  logs: string[];
  executionTime: number;
}

export const useRuntimeEngine = () => {
  const { toast } = useToast();
  const [isRunning, setIsRunning] = useState(false);
  const [result, setResult] = useState<RuntimeResult | null>(null);
  const iframeRef = useRef<HTMLIFrameElement | null>(null);

  const executeCode = useCallback(async (code: string, language: string = "javascript") => {
    setIsRunning(true);
    const startTime = performance.now();
    const logs: string[] = [];

    try {
      if (language === "javascript" || language === "typescript") {
        // Create sandboxed Web Worker for safer execution
        const workerCode = `
          let logs = [];
          self.console = {
            log: (...args) => {
              logs.push(args.map(a => String(a)).join(" "));
              self.postMessage({ type: 'log', data: logs });
            }
          };
          
          self.addEventListener('message', async (e) => {
            if (e.data.type === 'execute') {
              try {
                const AsyncFunction = Object.getPrototypeOf(async function(){}).constructor;
                const fn = new AsyncFunction(e.data.code);
                const result = await fn();
                self.postMessage({ 
                  type: 'result', 
                  data: String(result || 'Execution completed'),
                  logs: logs 
                });
              } catch (error) {
                self.postMessage({ 
                  type: 'error', 
                  error: error.message,
                  logs: logs 
                });
              }
            }
          });
        `;
        
        const blob = new Blob([workerCode], { type: 'application/javascript' });
        const workerUrl = URL.createObjectURL(blob);
        const worker = new Worker(workerUrl);
        
        // Set execution timeout
        const timeout = setTimeout(() => {
          worker.terminate();
          URL.revokeObjectURL(workerUrl);
          throw new Error('Execution timeout: Code took longer than 5 seconds');
        }, 5000);

        const result = await new Promise<{ output: string; logs: string[]; error?: string }>((resolve) => {
          worker.onmessage = (e) => {
            if (e.data.type === 'log') {
              // Update logs in real-time
            } else if (e.data.type === 'result') {
              clearTimeout(timeout);
              worker.terminate();
              URL.revokeObjectURL(workerUrl);
              resolve({ output: e.data.data, logs: e.data.logs || [] });
            } else if (e.data.type === 'error') {
              clearTimeout(timeout);
              worker.terminate();
              URL.revokeObjectURL(workerUrl);
              resolve({ output: '', logs: e.data.logs || [], error: e.data.error });
            }
          };
          
          worker.onerror = (error) => {
            clearTimeout(timeout);
            worker.terminate();
            URL.revokeObjectURL(workerUrl);
            resolve({ output: '', logs: [], error: error.message });
          };
          
          worker.postMessage({ type: 'execute', code });
        });

        if (result.error) {
          throw new Error(result.error);
        }

        const executionTime = performance.now() - startTime;

        setResult({
          output: result.output,
          logs: result.logs,
          executionTime,
        });

        toast({ title: "Code executed successfully" });
      } else {
        throw new Error(`Language ${language} not supported for browser execution`);
      }
    } catch (error: any) {
      const executionTime = performance.now() - startTime;
      
      setResult({
        output: "",
        error: error.message,
        logs,
        executionTime,
      });

      toast({
        title: "Execution Error",
        description: error.message,
        variant: "destructive",
      });
    } finally {
      setIsRunning(false);
    }
  }, [toast]);

  const clearResult = useCallback(() => {
    setResult(null);
  }, []);

  return {
    executeCode,
    clearResult,
    isRunning,
    result,
  };
};
