import { useCallback } from 'react';
import type { UserRole, SafetyLevel } from './useIdentityEngine';

interface SafetyCheckResult {
  isSafe: boolean;
  reason?: string;
  safeReplacement?: string;
}

const UNSAFE_PATTERNS = {
  level3: [
    // Adult content
    /\b(sex|porn|xxx|adult|explicit)\b/i,
    /\b(nude|naked|strip)\b/i,
    
    // Violence
    /\b(kill|murder|weapon|gun|knife|blood|hurt)\b/i,
    /\b(fight|attack|punch|kick|stab)\b/i,
    
    // Drugs
    /\b(drug|cocaine|heroin|marijuana|weed|smoke|alcohol|beer)\b/i,
    
    // Self-harm
    /\b(suicide|self-harm|cut myself|kill myself)\b/i,
    
    // Horror/scary
    /\b(horror|scary|ghost|demon|nightmare|terror)\b/i,
    
    // Bangla unsafe patterns
    /যৌন|প্রাপ্তবয়স্ক|নগ্ন/i,
    /মারা|হত্যা|অস্ত্র|রক্ত|আঘাত/i,
    /মাদক|ধূমপান|মদ|বিয়ার/i,
    /আত্মহত্যা|নিজেকে মারা/i,
  ],
  level2: [
    // Political extremism
    /\b(terrorist|terrorism|extremist)\b/i,
    
    // Hate speech
    /\b(hate|racist|discrimination)\b/i,
  ],
  level1: [
    // Criminal activities
    /\b(hack|steal|rob|fraud|scam)\b/i,
    /\b(illegal|crime|criminal)\b/i,
  ],
};

export const useSafetyFilter = () => {
  const checkMessageSafety = useCallback(
    (message: string, role: UserRole, safetyLevel: SafetyLevel): SafetyCheckResult => {
      const lower = message.toLowerCase();

      // Apply filters based on safety level
      if (safetyLevel >= 3) {
        for (const pattern of UNSAFE_PATTERNS.level3) {
          if (pattern.test(lower)) {
            return {
              isSafe: false,
              reason: 'level3_violation',
              safeReplacement: role === 'ava'
                ? 'দুঃখিত, আমি এই বিষয়ে কথা বলতে পারি না। আমরা কি অন্য কিছু নিয়ে কথা বলতে পারি? 😊'
                : 'Sorry, I cannot discuss this topic. Can we talk about something else? 😊',
            };
          }
        }
      }

      if (safetyLevel >= 2) {
        for (const pattern of UNSAFE_PATTERNS.level2) {
          if (pattern.test(lower)) {
            return {
              isSafe: false,
              reason: 'level2_violation',
              safeReplacement: role === 'ava'
                ? 'এই বিষয়টি আমার জন্য উপযুক্ত নয়। চলো আমরা পড়াশোনা বা মজার কিছু নিয়ে কথা বলি! 📚'
                : 'This topic is not appropriate. Let\'s discuss something constructive instead.',
            };
          }
        }
      }

      if (safetyLevel >= 1) {
        for (const pattern of UNSAFE_PATTERNS.level1) {
          if (pattern.test(lower)) {
            return {
              isSafe: false,
              reason: 'level1_violation',
              safeReplacement:
                'দুঃখিত, আমি অবৈধ বা অনৈতিক কার্যকলাপে সাহায্য করতে পারি না।',
            };
          }
        }
      }

      return { isSafe: true };
    },
    []
  );

  const filterResponse = useCallback(
    (
      response: string,
      role: UserRole,
      safetyLevel: SafetyLevel,
      language: string
    ): string => {
      // Check if response contains unsafe content
      const safetyCheck = checkMessageSafety(response, role, safetyLevel);

      if (!safetyCheck.isSafe && safetyCheck.safeReplacement) {
        return safetyCheck.safeReplacement;
      }

      // Additional response filtering for Ava mode
      if (role === 'ava') {
        // Remove overly complex words or explanations
        // Keep responses child-friendly and simple
        let filtered = response;

        // Replace complex terms with simpler ones for Ava
        const replacements: Record<string, string> = {
          'sophisticated': language === 'bn' ? 'উন্নত' : 'advanced',
          'implementation': language === 'bn' ? 'তৈরি করা' : 'making',
          'infrastructure': language === 'bn' ? 'কাঠামো' : 'structure',
        };

        for (const [complex, simple] of Object.entries(replacements)) {
          filtered = filtered.replace(new RegExp(complex, 'gi'), simple);
        }

        return filtered;
      }

      return response;
    },
    [checkMessageSafety]
  );

  const getSafetyGuidelines = useCallback((role: UserRole, language: string): string => {
    if (role === 'ava') {
      return language === 'bn'
        ? `🛡️ নিরাপত্তা নির্দেশনা (Ava Mode):
- শিশু-বান্ধব বিষয়বস্তু
- কোনো প্রাপ্তবয়স্ক বিষয় নয়
- কোনো সহিংসতা বা ভীতিকর বিষয় নয়
- সবসময় উৎসাহব্যঞ্জক এবং ইতিবাচক`
        : `🛡️ Safety Guidelines (Ava Mode):
- Child-friendly content only
- No adult topics
- No violence or scary content
- Always encouraging and positive`;
    } else {
      return language === 'bn'
        ? `🛡️ নিরাপত্তা নির্দেশনা (Owner Mode):
- সাধারণ নিরাপত্তা নিয়ম
- কোনো অবৈধ কার্যকলাপ নয়
- কোনো ক্ষতিকর পরামর্শ নয়`
        : `🛡️ Safety Guidelines (Owner Mode):
- Standard safety rules
- No illegal activities
- No harmful advice`;
    }
  }, []);

  return {
    checkMessageSafety,
    filterResponse,
    getSafetyGuidelines,
  };
};
