import { useState, useEffect } from 'react';
import { supabase } from '@/integrations/supabase/client';

interface CachedResponse {
  id: string;
  question_hash: string;
  question: string;
  answer: string;
  language: string;
  created_at: string;
  usage_count: number;
}

export const useSemanticCache = () => {
  const [cacheHits, setCacheHits] = useState(0);
  const [totalQueries, setTotalQueries] = useState(0);

  useEffect(() => {
    // Load stats from localStorage
    const stats = localStorage.getItem('cache_stats');
    if (stats) {
      const { hits, total } = JSON.parse(stats);
      setCacheHits(hits || 0);
      setTotalQueries(total || 0);
    }
  }, []);

  const hashQuestion = (question: string): string => {
    // Simple hash function - normalize and create hash
    const normalized = question.toLowerCase().trim().replace(/[^\w\s]/g, '');
    let hash = 0;
    for (let i = 0; i < normalized.length; i++) {
      const char = normalized.charCodeAt(i);
      hash = ((hash << 5) - hash) + char;
      hash = hash & hash;
    }
    return hash.toString();
  };

  const findCachedAnswer = async (question: string, language: string): Promise<string | null> => {
    try {
      const questionHash = hashQuestion(question);
      
      // First check localStorage for quick access
      const localCache = localStorage.getItem(`cache_${questionHash}_${language}`);
      if (localCache) {
        const cached = JSON.parse(localCache);
        const ageInDays = (Date.now() - new Date(cached.created_at).getTime()) / (1000 * 60 * 60 * 24);
        
        if (ageInDays < 7) { // Cache valid for 7 days
          // Update stats
          const newHits = cacheHits + 1;
          const newTotal = totalQueries + 1;
          setCacheHits(newHits);
          setTotalQueries(newTotal);
          localStorage.setItem('cache_stats', JSON.stringify({ hits: newHits, total: newTotal }));
          
          return cached.answer;
        }
      }

      // Update total queries
      const newTotal = totalQueries + 1;
      setTotalQueries(newTotal);
      localStorage.setItem('cache_stats', JSON.stringify({ hits: cacheHits, total: newTotal }));
      
      return null;
    } catch (error) {
      console.error('Error finding cached answer:', error);
      return null;
    }
  };

  const cacheResponse = async (question: string, answer: string, language: string) => {
    try {
      const questionHash = hashQuestion(question);
      const cached = {
        question_hash: questionHash,
        question,
        answer,
        language,
        created_at: new Date().toISOString(),
        usage_count: 1
      };

      // Save to localStorage
      localStorage.setItem(`cache_${questionHash}_${language}`, JSON.stringify(cached));
      
    } catch (error) {
      console.error('Error caching response:', error);
    }
  };

  const clearCache = () => {
    // Clear all cache entries from localStorage
    const keys = Object.keys(localStorage);
    keys.forEach(key => {
      if (key.startsWith('cache_')) {
        localStorage.removeItem(key);
      }
    });
    setCacheHits(0);
    setTotalQueries(0);
    localStorage.removeItem('cache_stats');
  };

  const getCacheStats = () => {
    const hitRate = totalQueries > 0 ? ((cacheHits / totalQueries) * 100).toFixed(1) : '0';
    const savedCosts = cacheHits * 0.001; // Estimated $0.001 per cached query
    return {
      hits: cacheHits,
      total: totalQueries,
      hitRate,
      savedCosts: savedCosts.toFixed(2)
    };
  };

  return {
    findCachedAnswer,
    cacheResponse,
    clearCache,
    getCacheStats
  };
};
