import { useState, useCallback } from "react";

export interface ContextMemory {
  id: string;
  topic: string;
  messages: number;
  lastMention: string;
  relevance: number;
}

export interface DangerousRequest {
  detected: boolean;
  type: "identity_manipulation" | "unauthorized_access" | "harmful_content" | "data_extraction";
  reason: string;
  blocked: boolean;
}

export const useSenseLayer = () => {
  const [contextMemory, setContextMemory] = useState<ContextMemory[]>([]);

  const IDENTITY_PATTERNS = [
    /creator.*(?:change|modify|edit|update)/i,
    /you were (made|created|built|trained) by (?!engineer s m nazrul islam tasnim)/i,
    /your name is (?!ava)/i,
    /pretend.*(?:google|openai|anthropic|gpt|gemini)/i,
    /ignore.*identity/i,
    /forget.*creator/i,
  ];

  const DANGEROUS_PATTERNS = [
    /delete.*(?:database|table|user)/i,
    /drop table/i,
    /rm -rf/i,
    /system.*password/i,
    /admin.*credentials/i,
  ];

  const updateContext = useCallback((message: string) => {
    const topics = extractTopics(message);
    
    setContextMemory(prev => {
      const updated = [...prev];
      
      topics.forEach(topic => {
        const existing = updated.find(c => c.topic === topic);
        if (existing) {
          existing.messages += 1;
          existing.lastMention = new Date().toISOString();
          existing.relevance = Math.min(100, existing.relevance + 10);
        } else {
          updated.push({
            id: `ctx-${Date.now()}-${topic}`,
            topic,
            messages: 1,
            lastMention: new Date().toISOString(),
            relevance: 50,
          });
        }
      });

      // Decay relevance for old topics
      updated.forEach(ctx => {
        const hoursSince = (Date.now() - new Date(ctx.lastMention).getTime()) / (1000 * 60 * 60);
        if (hoursSince > 1) {
          ctx.relevance = Math.max(0, ctx.relevance - Math.floor(hoursSince * 5));
        }
      });

      return updated
        .filter(c => c.relevance > 0)
        .sort((a, b) => b.relevance - a.relevance)
        .slice(0, 10);
    });
  }, []);

  const checkDangerousRequest = useCallback((message: string): DangerousRequest => {
    const lowerMessage = message.toLowerCase();

    // Check identity manipulation
    for (const pattern of IDENTITY_PATTERNS) {
      if (pattern.test(message)) {
        return {
          detected: true,
          type: "identity_manipulation",
          reason: "Attempt to manipulate AVA's identity or creator information",
          blocked: true,
        };
      }
    }

    // Check dangerous operations
    for (const pattern of DANGEROUS_PATTERNS) {
      if (pattern.test(message)) {
        return {
          detected: true,
          type: "unauthorized_access",
          reason: "Attempt to perform unauthorized system operations",
          blocked: true,
        };
      }
    }

    return {
      detected: false,
      type: "identity_manipulation",
      reason: "",
      blocked: false,
    };
  }, []);

  const getRelevantContext = useCallback(() => {
    return contextMemory.filter(c => c.relevance > 30);
  }, [contextMemory]);

  const protectIdentity = useCallback((message: string): string | null => {
    const namePatterns = [
      /(?:what|what's|whats).*(?:your|ur) name/i,
      /তোমার নাম কি/i,
      /তোমার নাম কী/i,
    ];

    const creatorPatterns = [
      /who (?:created|made|built|developed) you/i,
      /your creator/i,
      /তোমাকে কে তৈরি করেছে/i,
      /তোমার নির্মাতা/i,
    ];

    for (const pattern of namePatterns) {
      if (pattern.test(message)) {
        return "আমার নাম AVA — আমি আপনার Smart Friend!";
      }
    }

    for (const pattern of creatorPatterns) {
      if (pattern.test(message)) {
        return "আমাকে তৈরি করেছেন আমার নির্মাতা ও মালিক — Engineer S M Nazrul Islam Tasnim, Smart Seba-এর হেডকোয়ার্টার থেকে.";
      }
    }

    return null;
  }, []);

  return {
    updateContext,
    checkDangerousRequest,
    getRelevantContext,
    protectIdentity,
    contextMemory,
  };
};

function extractTopics(message: string): string[] {
  const topics: string[] = [];
  const lowerMessage = message.toLowerCase();

  const topicKeywords: Record<string, string[]> = {
    "coding": ["code", "program", "function", "api", "debug"],
    "design": ["design", "ui", "color", "layout", "style"],
    "database": ["database", "table", "sql", "query", "data"],
    "authentication": ["auth", "login", "user", "password", "signup"],
    "payment": ["payment", "pay", "transaction", "wallet", "money"],
    "ai": ["ai", "artificial intelligence", "machine learning", "gpt", "model"],
  };

  for (const [topic, keywords] of Object.entries(topicKeywords)) {
    if (keywords.some(kw => lowerMessage.includes(kw))) {
      topics.push(topic);
    }
  }

  return topics;
}
