import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";

export const useSubscription = () => {
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data: subscription, isLoading } = useQuery({
    queryKey: ["smart_subscription"],
    queryFn: async () => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      const { data, error } = await supabase
        .from("smart_subscriptions")
        .select("*")
        .eq("user_id", user.id)
        .single();

      if (error && error.code === "PGRST116") {
        return null;
      }

      if (error) throw error;
      return data;
    },
  });

  const activatePlan = useMutation({
    mutationFn: async (plan: string) => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      const expiresAt = new Date();
      expiresAt.setMonth(expiresAt.getMonth() + 1);

      const { data: existing } = await supabase
        .from("smart_subscriptions")
        .select("*")
        .eq("user_id", user.id)
        .single();

      if (existing) {
        const { error } = await supabase
          .from("smart_subscriptions")
          .update({
            plan,
            expires_at: expiresAt.toISOString(),
            is_active: true,
          })
          .eq("user_id", user.id);

        if (error) throw error;
      } else {
        const { error } = await supabase
          .from("smart_subscriptions")
          .insert({
            user_id: user.id,
            plan,
            expires_at: expiresAt.toISOString(),
            is_active: true,
          });

        if (error) throw error;
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["smart_subscription"] });
      toast({
        title: "Plan activated",
        description: "Your subscription is now active",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Error",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  return {
    subscription,
    isLoading,
    activatePlan,
  };
};
