import { useState, useCallback } from "react";
import { useToast } from "@/hooks/use-toast";
import { useAutoFix } from "@/hooks/useAutoFix";
import { useDeployment } from "@/hooks/useDeployment";
import { useErrorScanner } from "@/hooks/useErrorScanner";

export type TaskType = "build_module" | "fix_bug" | "refactor" | "deploy" | "optimize";
export type TaskStatus = "pending" | "running" | "completed" | "failed" | "cancelled";
export type TaskPriority = "low" | "medium" | "high" | "critical";

export interface TaskLog {
  timestamp: Date;
  level: "info" | "warning" | "error" | "success";
  message: string;
  details?: any;
}

export interface Task {
  id: string;
  title: string;
  description: string;
  type: TaskType;
  target: string; // project/module name
  priority: TaskPriority;
  status: TaskStatus;
  createdAt: Date;
  startedAt?: Date;
  completedAt?: Date;
  logs: TaskLog[];
  ownerApproved?: boolean;
  result?: {
    success: boolean;
    summary: string;
    details?: any;
  };
}

export const useTaskEngine = () => {
  const [tasks, setTasks] = useState<Task[]>([]);
  const [runningTask, setRunningTask] = useState<string | null>(null);
  const { toast } = useToast();
  const { generateFix, applyPatch } = useAutoFix();
  const { deploy } = useDeployment();
  const { scanProject } = useErrorScanner();

  const addLog = useCallback((taskId: string, log: Omit<TaskLog, "timestamp">) => {
    setTasks(prev => prev.map(task => {
      if (task.id === taskId) {
        return {
          ...task,
          logs: [...task.logs, { ...log, timestamp: new Date() }]
        };
      }
      return task;
    }));
  }, []);

  const updateTaskStatus = useCallback((taskId: string, status: TaskStatus) => {
    setTasks(prev => prev.map(task => {
      if (task.id === taskId) {
        const updates: Partial<Task> = { status };
        if (status === "running" && !task.startedAt) {
          updates.startedAt = new Date();
        }
        if (status === "completed" || status === "failed") {
          updates.completedAt = new Date();
        }
        return { ...task, ...updates };
      }
      return task;
    }));
  }, []);

  const createTask = useCallback((params: {
    title: string;
    description: string;
    type: TaskType;
    target: string;
    priority: TaskPriority;
  }): Task => {
    const task: Task = {
      id: `task_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
      ...params,
      status: "pending",
      createdAt: new Date(),
      logs: [],
    };

    setTasks(prev => [...prev, task]);
    
    toast({
      title: "Task Created",
      description: `${task.title} is ready to start`,
    });

    return task;
  }, [toast]);

  const approveTask = useCallback((taskId: string) => {
    setTasks(prev => prev.map(task => {
      if (task.id === taskId) {
        return { ...task, ownerApproved: true };
      }
      return task;
    }));
    
    addLog(taskId, {
      level: "success",
      message: "Task approved by Owner (Engineer S M Nazrul Islam Tasnim)",
    });
  }, [addLog]);

  const startTask = useCallback(async (taskId: string): Promise<boolean> => {
    const task = tasks.find(t => t.id === taskId);
    if (!task) {
      toast({ title: "Error", description: "Task not found", variant: "destructive" });
      return false;
    }

    if (task.priority === "critical" || task.type === "deploy") {
      if (!task.ownerApproved) {
        toast({
          title: "Owner Approval Required",
          description: "This task requires owner approval before execution",
          variant: "destructive",
        });
        return false;
      }
    }

    setRunningTask(taskId);
    updateTaskStatus(taskId, "running");
    addLog(taskId, { level: "info", message: `Starting task: ${task.title}` });

    try {
      let success = false;

      switch (task.type) {
        case "fix_bug":
          addLog(taskId, { level: "info", message: "Scanning for errors..." });
          const errors = await scanProject();
          
          if (errors.length > 0) {
            addLog(taskId, { level: "info", message: `Found ${errors.length} errors` });
            
            for (const error of errors.slice(0, 3)) {
              addLog(taskId, { 
                level: "warning", 
                message: `Fixing: ${error.message}`,
                details: error 
              });
              
              const patch = await generateFix(error);
              if (patch) {
                await applyPatch(patch.id);
              }
            }
            
            success = true;
            addLog(taskId, { level: "success", message: "Bug fixes applied" });
          } else {
            addLog(taskId, { level: "info", message: "No errors found" });
            success = true;
          }
          break;

        case "deploy":
          addLog(taskId, { level: "info", message: "Starting deployment pipeline..." });
          
          const deployResult = await deploy({
            method: "ftp",
            host: "auto-deploy.smartseba.com",
            username: "auto",
            password: "***",
            path: `/apps/${task.target}`,
          });
          
          success = deployResult;
          addLog(taskId, { 
            level: success ? "success" : "error", 
            message: success ? "Deployment successful" : "Deployment failed" 
          });
          break;

        case "build_module":
          addLog(taskId, { level: "info", message: "Building module structure..." });
          await new Promise(resolve => setTimeout(resolve, 2000));
          
          addLog(taskId, { level: "info", message: "Generating backend (Laravel)..." });
          await new Promise(resolve => setTimeout(resolve, 1500));
          
          addLog(taskId, { level: "info", message: "Generating frontend (React)..." });
          await new Promise(resolve => setTimeout(resolve, 1500));
          
          addLog(taskId, { level: "info", message: "Integrating auth and safety rules..." });
          await new Promise(resolve => setTimeout(resolve, 1000));
          
          success = true;
          addLog(taskId, { level: "success", message: "Module created successfully" });
          break;

        case "refactor":
          addLog(taskId, { level: "info", message: "Analyzing code structure..." });
          await new Promise(resolve => setTimeout(resolve, 1500));
          
          addLog(taskId, { level: "info", message: "Applying optimizations..." });
          await new Promise(resolve => setTimeout(resolve, 2000));
          
          success = true;
          addLog(taskId, { level: "success", message: "Refactoring completed" });
          break;

        case "optimize":
          addLog(taskId, { level: "info", message: "Running performance analysis..." });
          await new Promise(resolve => setTimeout(resolve, 1500));
          
          addLog(taskId, { level: "info", message: "Applying optimizations..." });
          await new Promise(resolve => setTimeout(resolve, 2000));
          
          success = true;
          addLog(taskId, { level: "success", message: "Optimization completed" });
          break;
      }

      if (success) {
        updateTaskStatus(taskId, "completed");
        setTasks(prev => prev.map(t => {
          if (t.id === taskId) {
            return {
              ...t,
              result: {
                success: true,
                summary: `${task.type} completed successfully`,
              }
            };
          }
          return t;
        }));
        
        toast({ 
          title: "Task Completed", 
          description: task.title 
        });
      } else {
        throw new Error("Task execution failed");
      }

      return true;
    } catch (error: any) {
      addLog(taskId, { 
        level: "error", 
        message: `Task failed: ${error.message}`,
        details: error 
      });
      
      updateTaskStatus(taskId, "failed");
      setTasks(prev => prev.map(t => {
        if (t.id === taskId) {
          return {
            ...t,
            result: {
              success: false,
              summary: error.message,
            }
          };
        }
        return t;
      }));
      
      toast({ 
        title: "Task Failed", 
        description: error.message,
        variant: "destructive" 
      });
      
      return false;
    } finally {
      setRunningTask(null);
    }
  }, [tasks, toast, updateTaskStatus, addLog, generateFix, applyPatch, deploy, scanProject]);

  const cancelTask = useCallback((taskId: string) => {
    updateTaskStatus(taskId, "cancelled");
    addLog(taskId, { level: "warning", message: "Task cancelled by user" });
    
    if (runningTask === taskId) {
      setRunningTask(null);
    }
    
    toast({ title: "Task Cancelled" });
  }, [runningTask, updateTaskStatus, addLog, toast]);

  const getTaskLog = useCallback((taskId: string): TaskLog[] => {
    const task = tasks.find(t => t.id === taskId);
    return task?.logs || [];
  }, [tasks]);

  const getTasksByStatus = useCallback((status: TaskStatus) => {
    return tasks.filter(t => t.status === status);
  }, [tasks]);

  const getSuggestedTasks = useCallback((): Array<{ title: string; description: string; type: TaskType; priority: TaskPriority }> => {
    return [
      {
        title: "Optimize Smart Marketer Page Speed",
        description: "Analyze and optimize loading times for Smart Marketer module",
        type: "optimize",
        priority: "medium",
      },
      {
        title: "Build Smart Academy Exam Engine",
        description: "Create complete exam system with AI grading",
        type: "build_module",
        priority: "high",
      },
      {
        title: "Refactor Smart Bazar Checkout",
        description: "Improve checkout flow and reduce code complexity",
        type: "refactor",
        priority: "medium",
      },
      {
        title: "Fix Smart Design Hub Export",
        description: "Resolve issues with design export functionality",
        type: "fix_bug",
        priority: "high",
      },
    ];
  }, []);

  return {
    tasks,
    runningTask,
    createTask,
    startTask,
    cancelTask,
    approveTask,
    getTaskLog,
    getTasksByStatus,
    getSuggestedTasks,
  };
};
