import { useState } from "react";
import { useLanguage } from "./useLanguage";

export const useTextToSpeech = () => {
  const [isSpeaking, setIsSpeaking] = useState(false);
  const { voiceCode } = useLanguage();

  const speak = (text: string, customLang?: string) => {
    if ("speechSynthesis" in window) {
      // Cancel any ongoing speech
      window.speechSynthesis.cancel();
      
      const utterance = new SpeechSynthesisUtterance(text);
      utterance.lang = customLang || voiceCode;
      
      // Adjust rate and pitch based on language
      const langCode = (customLang || voiceCode).split('-')[0];
      
      // Slower rate for complex scripts
      if (['ar', 'ur', 'fa', 'he', 'bn', 'hi', 'ta', 'te'].includes(langCode)) {
        utterance.rate = 0.85;
      } else if (['zh', 'ja', 'ko'].includes(langCode)) {
        utterance.rate = 0.9;
      } else {
        utterance.rate = 0.95;
      }
      
      utterance.pitch = 1;
      utterance.volume = 1;

      utterance.onstart = () => setIsSpeaking(true);
      utterance.onend = () => setIsSpeaking(false);
      utterance.onerror = () => setIsSpeaking(false);

      // Get available voices and select the best one for the language
      const voices = window.speechSynthesis.getVoices();
      const targetLang = (customLang || voiceCode).split('-')[0];
      const voice = voices.find(v => v.lang.startsWith(targetLang)) || voices[0];
      
      if (voice) {
        utterance.voice = voice;
      }

      window.speechSynthesis.speak(utterance);
    }
  };

  const stop = () => {
    if ("speechSynthesis" in window) {
      window.speechSynthesis.cancel();
      setIsSpeaking(false);
    }
  };

  return { isSpeaking, speak, stop };
};
