import { useState, useRef, useCallback } from "react";
import { supabase } from "@/integrations/supabase/client";
import { toast } from "sonner";

export interface CallPeer {
  id: string;
  name: string;
  avatar?: string;
}

export type CallStatus = "idle" | "calling" | "ringing" | "connected" | "ended";

export const useVoiceCall = () => {
  const [status, setStatus] = useState<CallStatus>("idle");
  const [peer, setPeer] = useState<CallPeer | null>(null);
  const [duration, setDuration] = useState(0);
  const [muted, setMuted] = useState(false);
  
  const pcRef = useRef<RTCPeerConnection | null>(null);
  const localStreamRef = useRef<MediaStream | null>(null);
  const remoteStreamRef = useRef<MediaStream | null>(null);
  const durationIntervalRef = useRef<number | null>(null);

  const initializePeerConnection = useCallback(async () => {
    try {
      // Get user media
      const stream = await navigator.mediaDevices.getUserMedia({
        audio: {
          echoCancellation: true,
          noiseSuppression: true,
          autoGainControl: true
        }
      });
      
      localStreamRef.current = stream;

      // Create peer connection
      pcRef.current = new RTCPeerConnection({
        iceServers: [
          { urls: "stun:stun.l.google.com:19302" },
          { urls: "stun:stun1.l.google.com:19302" }
        ]
      });

      // Add tracks
      stream.getTracks().forEach(track => {
        pcRef.current?.addTrack(track, stream);
      });

      // Handle remote stream
      pcRef.current.ontrack = (event) => {
        console.log("Received remote track");
        remoteStreamRef.current = event.streams[0];
      };

      // Handle ICE candidates
      pcRef.current.onicecandidate = (event) => {
        if (event.candidate) {
          console.log("New ICE candidate:", event.candidate);
          // TODO: Send to signaling server when implemented
        }
      };

      return true;
    } catch (error: any) {
      console.error("Failed to initialize peer connection:", error);
      toast.error(`Microphone access required: ${error.message}`);
      return false;
    }
  }, []);

  const startCall = useCallback(async (targetPeer: CallPeer) => {
    setStatus("calling");
    setPeer(targetPeer);
    toast.info(`Calling ${targetPeer.name}...`);

    const initialized = await initializePeerConnection();
    if (!initialized) {
      setStatus("idle");
      return;
    }

    try {
      // Create offer
      const offer = await pcRef.current?.createOffer();
      await pcRef.current?.setLocalDescription(offer);

      // Send offer through signaling
      const { error } = await supabase.from("calls").insert({
        caller_id: (await supabase.auth.getUser()).data.user?.id,
        callee_id: targetPeer.id,
        call_type: "voice",
        status: "ringing"
      });

      if (error) throw error;

      // Wait for answer...
      setStatus("ringing");
      
    } catch (error: any) {
      console.error("Failed to start call:", error);
      toast.error("Call failed");
      setStatus("idle");
    }
  }, [initializePeerConnection]);

  const answerCall = useCallback(async (callId: string, callerPeer: CallPeer) => {
    setPeer(callerPeer);
    setStatus("calling");
    toast.info(`Accepting call from ${callerPeer.name}...`);

    const initialized = await initializePeerConnection();
    if (!initialized) {
      setStatus("idle");
      return;
    }

    try {
      // Create answer
      const answer = await pcRef.current?.createAnswer();
      await pcRef.current?.setLocalDescription(answer);

      // Update call status
      const { error } = await supabase
        .from("calls")
        .update({ status: "active", started_at: new Date().toISOString() })
        .eq("id", callId);

      if (error) throw error;

      setStatus("connected");
      startDurationCounter();
      toast.success("Call connected");
      
    } catch (error: any) {
      console.error("Failed to answer call:", error);
      toast.error("Failed to answer call");
      setStatus("idle");
    }
  }, [initializePeerConnection]);

  const endCall = useCallback(async () => {
    // Stop all tracks
    localStreamRef.current?.getTracks().forEach(track => track.stop());
    remoteStreamRef.current?.getTracks().forEach(track => track.stop());

    // Close peer connection
    pcRef.current?.close();

    // Stop duration counter
    if (durationIntervalRef.current) {
      clearInterval(durationIntervalRef.current);
    }

    // Save call record
    if (status === "connected" && duration > 0) {
      const { data: { user } } = await supabase.auth.getUser();
      if (user) {
        await supabase.from("call_history").insert({
          user_id: user.id,
          call_duration: duration,
          ai_summary: `Voice call with ${peer?.name || "Unknown"}`
        });
      }
    }

    setStatus("ended");
    toast.info("Call ended");

    // Reset state
    setTimeout(() => {
      setStatus("idle");
      setPeer(null);
      setDuration(0);
    }, 2000);
  }, [status, duration, peer]);

  const toggleMute = useCallback(() => {
    if (!localStreamRef.current) return;

    const audioTrack = localStreamRef.current.getAudioTracks()[0];
    if (audioTrack) {
      audioTrack.enabled = !audioTrack.enabled;
      setMuted(!audioTrack.enabled);
      toast.info(audioTrack.enabled ? "Unmuted" : "Muted");
    }
  }, []);

  const startDurationCounter = () => {
    durationIntervalRef.current = window.setInterval(() => {
      setDuration(prev => prev + 1);
    }, 1000);
  };

  const getRemoteStream = useCallback(() => {
    return remoteStreamRef.current;
  }, []);

  return {
    startCall,
    answerCall,
    endCall,
    toggleMute,
    getRemoteStream,
    status,
    peer,
    duration,
    muted
  };
};
