import { useState, useCallback } from "react";
import { voiceInput } from "@/core/voice/voice.input";
import { voiceOutput, VoiceProfile } from "@/core/voice/voice.output";
import { useToast } from "@/hooks/use-toast";

export const useVoiceSystem = () => {
  const [isListening, setIsListening] = useState(false);
  const [isSpeaking, setIsSpeaking] = useState(false);
  const [voiceEnabled, setVoiceEnabled] = useState(false);
  const [voiceProfile, setVoiceProfile] = useState<VoiceProfile>("neutral");
  const { toast } = useToast();

  const startListening = useCallback(
    (onText: (text: string) => void) => {
      if (!voiceInput.isSupported()) {
        toast({
          title: "Voice Not Supported",
          description: "Your browser doesn't support voice input",
          variant: "destructive",
        });
        return;
      }

      voiceInput.startListening(
        (text) => {
          setIsListening(false);
          onText(text);
        },
        (error) => {
          setIsListening(false);
          toast({
            title: "Voice Error",
            description: error,
            variant: "destructive",
          });
        }
      );

      setIsListening(true);
    },
    [toast]
  );

  const stopListening = useCallback(() => {
    voiceInput.stopListening();
    setIsListening(false);
  }, []);

  const speak = useCallback(
    (text: string) => {
      if (!voiceOutput.isSupported()) {
        toast({
          title: "Voice Not Supported",
          description: "Your browser doesn't support text-to-speech",
          variant: "destructive",
        });
        return;
      }

      if (!voiceEnabled) return;

      voiceOutput.speak(text, voiceProfile);
      setIsSpeaking(true);

      // Update speaking state after a delay
      setTimeout(() => {
        setIsSpeaking(voiceOutput.getIsSpeaking());
      }, 1000);
    },
    [voiceEnabled, voiceProfile, toast]
  );

  const stopSpeaking = useCallback(() => {
    voiceOutput.stop();
    setIsSpeaking(false);
  }, []);

  const toggleVoice = useCallback(() => {
    setVoiceEnabled((prev) => !prev);
  }, []);

  const changeVoiceProfile = useCallback((profile: VoiceProfile) => {
    setVoiceProfile(profile);
  }, []);

  return {
    isListening,
    isSpeaking,
    voiceEnabled,
    voiceProfile,
    startListening,
    stopListening,
    speak,
    stopSpeaking,
    toggleVoice,
    changeVoiceProfile,
  };
};
