import { useState } from "react";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";
import { useLanguage } from "@/hooks/useLanguage";

export const useVoiceToText = () => {
  const [isRecording, setIsRecording] = useState(false);
  const [isProcessing, setIsProcessing] = useState(false);
  const { voiceCode } = useLanguage();
  const { toast } = useToast();

  const startRecording = async (): Promise<MediaRecorder | null> => {
    try {
      const stream = await navigator.mediaDevices.getUserMedia({ audio: true });
      const mediaRecorder = new MediaRecorder(stream);
      const chunks: Blob[] = [];

      mediaRecorder.ondataavailable = (e) => {
        if (e.data.size > 0) {
          chunks.push(e.data);
        }
      };

      mediaRecorder.onstop = async () => {
        setIsProcessing(true);
        const audioBlob = new Blob(chunks, { type: "audio/webm" });
        
        // Convert blob to base64
        const reader = new FileReader();
        reader.readAsDataURL(audioBlob);
        reader.onloadend = async () => {
          const base64Audio = reader.result?.toString().split(",")[1];
          
          try {
            const { data, error } = await supabase.functions.invoke("voice-to-text", {
              body: { audio: base64Audio },
            });

            if (error) throw error;
            
            // Emit custom event with transcribed text
            const event = new CustomEvent("voiceTranscribed", { detail: data.text });
            window.dispatchEvent(event);
            
            toast({ title: "Voice transcribed successfully" });
          } catch (error) {
            console.error("Voice to text error:", error);
            toast({
              title: "Transcription failed",
              description: error instanceof Error ? error.message : "Unknown error",
              variant: "destructive",
            });
          } finally {
            setIsProcessing(false);
          }
        };

        stream.getTracks().forEach((track) => track.stop());
      };

      setIsRecording(true);
      mediaRecorder.start();
      return mediaRecorder;
    } catch (error) {
      console.error("Error accessing microphone:", error);
      toast({
        title: "Microphone access denied",
        description: "Please allow microphone access to use voice input",
        variant: "destructive",
      });
      return null;
    }
  };

  return {
    isRecording,
    isProcessing,
    startRecording,
  };
};
