import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";

export const useWallet = () => {
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data: wallet, isLoading } = useQuery({
    queryKey: ["smart_wallet"],
    queryFn: async () => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      let { data, error } = await supabase
        .from("smart_wallets")
        .select("*")
        .eq("user_id", user.id)
        .single();

      if (error && error.code === "PGRST116") {
        const { data: newWallet, error: insertError } = await supabase
          .from("smart_wallets")
          .insert({ user_id: user.id, balance: 0 })
          .select()
          .single();

        if (insertError) throw insertError;
        return newWallet;
      }

      if (error) throw error;
      return data;
    },
  });

  const { data: transactions } = useQuery({
    queryKey: ["wallet_transactions"],
    queryFn: async () => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      const { data, error } = await supabase
        .from("wallet_transactions")
        .select("*")
        .eq("user_id", user.id)
        .order("created_at", { ascending: false })
        .limit(10);

      if (error) throw error;
      return data;
    },
  });

  const addBalance = useMutation({
    mutationFn: async (amount: number) => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      const { error: walletError } = await supabase
        .from("smart_wallets")
        .update({ balance: (wallet?.balance || 0) + amount })
        .eq("user_id", user.id);

      if (walletError) throw walletError;

      const { error: txError } = await supabase
        .from("wallet_transactions")
        .insert({
          user_id: user.id,
          amount,
          type: "credit",
          description: "Balance added",
        });

      if (txError) throw txError;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["smart_wallet"] });
      queryClient.invalidateQueries({ queryKey: ["wallet_transactions"] });
      toast({
        title: "Balance added",
        description: "Your balance has been updated",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Error",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  return {
    wallet,
    transactions,
    isLoading,
    addBalance,
  };
};
