import { useEffect, useRef, useState } from "react";
import { supabase } from "@/integrations/supabase/client";
import { toast } from "sonner";

const ICE_SERVERS = {
  iceServers: [
    { urls: "stun:stun.l.google.com:19302" },
    { urls: "stun:stun1.l.google.com:19302" },
  ],
};

type CallState = "idle" | "calling" | "ringing" | "connected" | "ended";

export const useWebRTC = (callId: string | null, isInitiator: boolean) => {
  const [callState, setCallState] = useState<CallState>("idle");
  const [remoteStream, setRemoteStream] = useState<MediaStream | null>(null);
  const [localStream, setLocalStream] = useState<MediaStream | null>(null);
  const peerConnection = useRef<RTCPeerConnection | null>(null);
  const signalChannel = useRef<any>(null);

  useEffect(() => {
    if (!callId) return;

    const setupWebRTC = async () => {
      try {
        // Get local media stream
        const stream = await navigator.mediaDevices.getUserMedia({
          video: true,
          audio: true,
        });
        setLocalStream(stream);

        // Create peer connection
        peerConnection.current = new RTCPeerConnection(ICE_SERVERS);

        // Add local stream tracks
        stream.getTracks().forEach((track) => {
          peerConnection.current?.addTrack(track, stream);
        });

        // Handle remote stream
        peerConnection.current.ontrack = (event) => {
          setRemoteStream(event.streams[0]);
          setCallState("connected");
        };

        // Handle ICE candidates
        peerConnection.current.onicecandidate = async (event) => {
          if (event.candidate) {
            const { data: { user } } = await supabase.auth.getUser();
            if (!user) return;

            // Send ICE candidate through signaling
            await supabase.from("webrtc_signals").insert({
              call_id: callId,
              from_user_id: user.id,
              to_user_id: isInitiator ? "" : user.id,
              signal_type: "ice-candidate",
              signal_data: event.candidate.toJSON() as any,
            } as any);
          }
        };

        // Subscribe to signaling messages
        const { data: { user } } = await supabase.auth.getUser();
        if (!user) return;

        signalChannel.current = supabase
          .channel(`webrtc:${callId}`)
          .on(
            "postgres_changes",
            {
              event: "INSERT",
              schema: "public",
              table: "webrtc_signals",
              filter: `call_id=eq.${callId}`,
            },
            async (payload) => {
              const signal = payload.new;
              if (signal.from_user_id === user.id) return;

              if (signal.signal_type === "offer") {
                await peerConnection.current?.setRemoteDescription(
                  new RTCSessionDescription(signal.signal_data)
                );
                const answer = await peerConnection.current?.createAnswer();
                await peerConnection.current?.setLocalDescription(answer);
                
                await supabase.from("webrtc_signals").insert({
                  call_id: callId,
                  from_user_id: user.id,
                  to_user_id: signal.from_user_id,
                  signal_type: "answer",
                  signal_data: answer as any,
                } as any);
              } else if (signal.signal_type === "answer") {
                await peerConnection.current?.setRemoteDescription(
                  new RTCSessionDescription(signal.signal_data)
                );
              } else if (signal.signal_type === "ice-candidate") {
                await peerConnection.current?.addIceCandidate(
                  new RTCIceCandidate(signal.signal_data)
                );
              }
            }
          )
          .subscribe();

        // If initiator, create offer
        if (isInitiator) {
          setCallState("calling");
          const offer = await peerConnection.current.createOffer();
          await peerConnection.current.setLocalDescription(offer);

          await supabase.from("webrtc_signals").insert({
            call_id: callId,
            from_user_id: user.id,
            to_user_id: "",
            signal_type: "offer",
            signal_data: offer as any,
          } as any);
        } else {
          setCallState("ringing");
        }
      } catch (error) {
        console.error("WebRTC setup error:", error);
        toast.error("Failed to setup call");
        setCallState("ended");
      }
    };

    setupWebRTC();

    return () => {
      signalChannel.current?.unsubscribe();
      localStream?.getTracks().forEach((track) => track.stop());
      peerConnection.current?.close();
    };
  }, [callId, isInitiator]);

  const endCall = async () => {
    if (!callId) return;

    const endTime = new Date().toISOString();
    
    await supabase
      .from("calls")
      .update({ 
        status: "ended", 
        ended_at: endTime
      })
      .eq("id", callId);

    localStream?.getTracks().forEach((track) => track.stop());
    peerConnection.current?.close();
    setCallState("ended");
  };

  const toggleVideo = () => {
    if (localStream) {
      const videoTrack = localStream.getVideoTracks()[0];
      if (videoTrack) {
        videoTrack.enabled = !videoTrack.enabled;
      }
    }
  };

  const toggleAudio = () => {
    if (localStream) {
      const audioTrack = localStream.getAudioTracks()[0];
      if (audioTrack) {
        audioTrack.enabled = !audioTrack.enabled;
      }
    }
  };

  return {
    callState,
    localStream,
    remoteStream,
    endCall,
    toggleVideo,
    toggleAudio,
  };
};
