import { useState } from "react";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";
import { useLanguage } from "./useLanguage";
import type { SearchResult, SearchResponse, SearchOptions } from "@/types/search";

export const useWebSearch = () => {
  const [searching, setSearching] = useState(false);
  const [results, setResults] = useState<SearchResult[]>([]);
  const [cachedResults, setCachedResults] = useState<Map<string, SearchResult[]>>(new Map());
  const { toast } = useToast();
  const { language } = useLanguage();

  const search = async (options: SearchOptions): Promise<SearchResult[]> => {
    setSearching(true);
    
    // Check cache first
    const cacheKey = `${options.query}-${options.lang}-${options.maxResults}`;
    if (cachedResults.has(cacheKey)) {
      const cached = cachedResults.get(cacheKey)!;
      setResults(cached);
      setSearching(false);
      return cached;
    }

    try {
      const { data, error } = await supabase.functions.invoke('web-search', {
        body: {
          query: options.query,
          lang: options.lang || language,
          maxResults: options.maxResults || 5,
          module: options.module,
          safeSearch: options.safeSearch !== false,
        }
      });

      if (error) throw error;

      const searchResponse = data as SearchResponse;
      const searchResults = searchResponse.results || [];
      
      setResults(searchResults);
      
      // Cache results
      const newCache = new Map(cachedResults);
      newCache.set(cacheKey, searchResults);
      setCachedResults(newCache);

      // Store in localStorage for offline access
      try {
        localStorage.setItem(`search_cache_${cacheKey}`, JSON.stringify({
          results: searchResults,
          timestamp: Date.now(),
        }));
      } catch (e) {
        console.warn('Failed to cache search results', e);
      }

      return searchResults;
    } catch (error) {
      console.error('Search error:', error);
      
      // Try offline cache
      try {
        const offlineCache = localStorage.getItem(`search_cache_${cacheKey}`);
        if (offlineCache) {
          const cached = JSON.parse(offlineCache);
          const age = Date.now() - cached.timestamp;
          
          // Use cache if less than 24 hours old
          if (age < 24 * 60 * 60 * 1000) {
            setResults(cached.results);
            toast({
              title: "Offline Mode",
              description: "Showing cached results",
            });
            return cached.results;
          }
        }
      } catch (e) {
        console.error('Failed to load offline cache', e);
      }

      toast({
        title: "Search Failed",
        description: error instanceof Error ? error.message : "Unknown error",
        variant: "destructive",
      });
      
      return [];
    } finally {
      setSearching(false);
    }
  };

  const clearCache = () => {
    setCachedResults(new Map());
    // Clear localStorage cache
    Object.keys(localStorage).forEach(key => {
      if (key.startsWith('search_cache_')) {
        localStorage.removeItem(key);
      }
    });
    toast({
      title: "Cache Cleared",
      description: "Search cache has been cleared",
    });
  };

  return {
    search,
    searching,
    results,
    clearCache,
  };
};
