import { useCallback } from "react";
import { useToast } from "@/hooks/use-toast";
import { useProjectFiles } from "./useProjectFiles";

export const useZipExport = (projectId: string) => {
  const { files } = useProjectFiles(projectId);
  const { toast } = useToast();

  const exportToZip = useCallback(async (projectName: string = "smart-friend-project") => {
    try {
      // Dynamic import JSZip
      const JSZip = (await import("jszip")).default;
      const zip = new JSZip();

      // Add all files to zip
      files.forEach((file) => {
        if (file.file_type !== "folder" && file.content) {
          zip.file(file.file_path, file.content);
        }
      });

      // Add package.json
      const packageJson = {
        name: projectName,
        version: "1.0.0",
        description: "Smart Friend generated project",
        scripts: {
          dev: "vite",
          build: "vite build",
          preview: "vite preview"
        },
        dependencies: {
          "react": "^18.3.1",
          "react-dom": "^18.3.1"
        },
        devDependencies: {
          "vite": "^5.0.0",
          "@vitejs/plugin-react": "^4.2.0",
          "typescript": "^5.3.0"
        }
      };

      zip.file("package.json", JSON.stringify(packageJson, null, 2));

      // Add README
      const readme = `# ${projectName}

Generated by Smart Friend Ultra v5.0

## Installation

\`\`\`bash
npm install
\`\`\`

## Development

\`\`\`bash
npm run dev
\`\`\`

## Build

\`\`\`bash
npm run build
\`\`\`
`;

      zip.file("README.md", readme);

      // Generate zip
      const blob = await zip.generateAsync({ type: "blob" });

      // Download
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = `${projectName}.zip`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);

      toast({ title: "Project exported successfully" });
    } catch (error: any) {
      toast({
        title: "Export Error",
        description: error.message,
        variant: "destructive",
      });
    }
  }, [files, toast]);

  return { exportToZip };
};
