export interface LaravelRuntimeConfig {
  phpVersion: "8.1" | "8.2" | "8.3";
  composerPath: string;
  runMigrations: boolean;
  runSeed: boolean;
  runOptimize: boolean;
  forceComposerInstall: boolean;
  defaultTimeoutSeconds: number;
  defaultArtisanCommands: string[];
}

export const defaultLaravelConfig: LaravelRuntimeConfig = {
  phpVersion: "8.2",
  composerPath: "composer",
  runMigrations: true,
  runSeed: false,
  runOptimize: true,
  forceComposerInstall: false,
  defaultTimeoutSeconds: 600,
  defaultArtisanCommands: [
    "php artisan key:generate --force",
    "php artisan migrate --force",
    "php artisan optimize"
  ]
};

export const getLaravelConfig = (): LaravelRuntimeConfig => {
  const stored = localStorage.getItem('laravel_runtime_config');
  if (stored) {
    try {
      return { ...defaultLaravelConfig, ...JSON.parse(stored) };
    } catch (e) {
      console.error('Failed to parse Laravel config:', e);
    }
  }
  return defaultLaravelConfig;
};

export const saveLaravelConfig = (config: Partial<LaravelRuntimeConfig>) => {
  const current = getLaravelConfig();
  const updated = { ...current, ...config };
  localStorage.setItem('laravel_runtime_config', JSON.stringify(updated));
  return updated;
};

export const getArtisanCommands = (config: LaravelRuntimeConfig): string[] => {
  const commands: string[] = [];
  
  // Always generate key first if needed
  commands.push("php artisan key:generate --force");
  
  if (config.runMigrations) {
    commands.push("php artisan migrate --force");
  }
  
  if (config.runSeed) {
    commands.push("php artisan db:seed --force");
  }
  
  if (config.runOptimize) {
    commands.push("php artisan config:cache");
    commands.push("php artisan route:cache");
    commands.push("php artisan view:cache");
    commands.push("php artisan optimize");
  }
  
  return commands;
};
