export interface CommandTarget {
  type: "ssh" | "cpanel" | "http";
  host: string;
  port?: number;
  username?: string;
  password?: string;
  privateKey?: string;
  apiToken?: string;
  basePath?: string;
}

export interface CommandResult {
  success: boolean;
  stdout: string;
  stderr: string;
  exitCode: number;
  duration: number;
}

export class RemoteCommandRunner {
  private target: CommandTarget;
  private logs: Array<{ timestamp: string; message: string; level: string }> = [];

  constructor(target: CommandTarget) {
    this.target = target;
  }

  private log(level: string, message: string) {
    const entry = {
      timestamp: new Date().toISOString(),
      message,
      level
    };
    this.logs.push(entry);
    console.log(`[${level.toUpperCase()}] ${message}`);
  }

  getLogs() {
    return [...this.logs];
  }

  clearLogs() {
    this.logs = [];
  }

  async runSSHCommand(command: string): Promise<CommandResult> {
    this.log("info", `SSH: Executing command: ${command}`);
    const startTime = Date.now();

    try {
      // Simulate SSH command execution
      await new Promise(resolve => setTimeout(resolve, 1500));
      
      const mockOutput = `Command executed: ${command}\nCompleted successfully`;
      const duration = Date.now() - startTime;
      
      this.log("success", `SSH: Command completed in ${duration}ms`);
      
      return {
        success: true,
        stdout: mockOutput,
        stderr: "",
        exitCode: 0,
        duration
      };
    } catch (error: any) {
      const duration = Date.now() - startTime;
      this.log("error", `SSH: Command failed: ${error.message}`);
      
      return {
        success: false,
        stdout: "",
        stderr: error.message,
        exitCode: 1,
        duration
      };
    }
  }

  async runCPanelCommand(command: string): Promise<CommandResult> {
    this.log("info", `cPanel: Executing command: ${command}`);
    const startTime = Date.now();

    try {
      // Simulate cPanel terminal API call
      await new Promise(resolve => setTimeout(resolve, 2000));
      
      const mockOutput = `cPanel execution: ${command}\nOutput: Success`;
      const duration = Date.now() - startTime;
      
      this.log("success", `cPanel: Command completed in ${duration}ms`);
      
      return {
        success: true,
        stdout: mockOutput,
        stderr: "",
        exitCode: 0,
        duration
      };
    } catch (error: any) {
      const duration = Date.now() - startTime;
      this.log("error", `cPanel: Command failed: ${error.message}`);
      
      return {
        success: false,
        stdout: "",
        stderr: error.message,
        exitCode: 1,
        duration
      };
    }
  }

  async runArtisanCommand(artisanCommand: string): Promise<CommandResult> {
    // Validate that this is a safe artisan command
    const whitelistedCommands = [
      "key:generate",
      "migrate",
      "db:seed",
      "optimize",
      "config:cache",
      "route:cache",
      "view:cache",
      "cache:clear",
      "config:clear",
      "route:clear",
      "view:clear"
    ];

    const commandBase = artisanCommand.replace(/^php\s+artisan\s+/, "").split(" ")[0];
    
    if (!whitelistedCommands.includes(commandBase)) {
      this.log("error", `Artisan: Command not whitelisted: ${commandBase}`);
      return {
        success: false,
        stdout: "",
        stderr: `Command not allowed: ${commandBase}`,
        exitCode: 1,
        duration: 0
      };
    }

    this.log("info", `Artisan: Executing ${artisanCommand}`);
    
    switch (this.target.type) {
      case "ssh":
        return this.runSSHCommand(artisanCommand);
      case "cpanel":
        return this.runCPanelCommand(artisanCommand);
      case "http":
        return this.runHTTPCommand(artisanCommand);
      default:
        throw new Error(`Unsupported target type: ${this.target.type}`);
    }
  }

  private async runHTTPCommand(command: string): Promise<CommandResult> {
    this.log("info", `HTTP: Executing via artisan-runner.php: ${command}`);
    const startTime = Date.now();

    try {
      // Simulate HTTP-based artisan runner
      await new Promise(resolve => setTimeout(resolve, 2500));
      
      const mockOutput = `HTTP Runner executed: ${command}\nStatus: OK`;
      const duration = Date.now() - startTime;
      
      this.log("success", `HTTP: Command completed in ${duration}ms`);
      
      return {
        success: true,
        stdout: mockOutput,
        stderr: "",
        exitCode: 0,
        duration
      };
    } catch (error: any) {
      const duration = Date.now() - startTime;
      this.log("error", `HTTP: Command failed: ${error.message}`);
      
      return {
        success: false,
        stdout: "",
        stderr: error.message,
        exitCode: 1,
        duration
      };
    }
  }
}

export const createCommandRunner = (target: CommandTarget): RemoteCommandRunner => {
  return new RemoteCommandRunner(target);
};
