// Comprehensive language support for Smart Seba
// 100+ languages with metadata

export type LanguageCode = string;

export interface Language {
  code: string;
  name: string;
  nativeName: string;
  direction: 'ltr' | 'rtl';
  voiceCode: string; // For TTS/STT
  region: string;
  popular?: boolean; // Show in quick switcher
}

export const LANGUAGES: Record<string, Language> = {
  // Popular languages (shown first)
  en: { code: 'en', name: 'English', nativeName: 'English', direction: 'ltr', voiceCode: 'en-US', region: 'Global', popular: true },
  bn: { code: 'bn', name: 'Bengali', nativeName: 'বাংলা', direction: 'ltr', voiceCode: 'bn-BD', region: 'Bangladesh', popular: true },
  ar: { code: 'ar', name: 'Arabic', nativeName: 'العربية', direction: 'rtl', voiceCode: 'ar-SA', region: 'Middle East', popular: true },
  hi: { code: 'hi', name: 'Hindi', nativeName: 'हिन्दी', direction: 'ltr', voiceCode: 'hi-IN', region: 'India', popular: true },
  ur: { code: 'ur', name: 'Urdu', nativeName: 'اردو', direction: 'rtl', voiceCode: 'ur-PK', region: 'Pakistan', popular: true },
  es: { code: 'es', name: 'Spanish', nativeName: 'Español', direction: 'ltr', voiceCode: 'es-ES', region: 'Spain', popular: true },
  fr: { code: 'fr', name: 'French', nativeName: 'Français', direction: 'ltr', voiceCode: 'fr-FR', region: 'France', popular: true },
  de: { code: 'de', name: 'German', nativeName: 'Deutsch', direction: 'ltr', voiceCode: 'de-DE', region: 'Germany', popular: true },
  zh: { code: 'zh', name: 'Chinese', nativeName: '中文', direction: 'ltr', voiceCode: 'zh-CN', region: 'China', popular: true },
  ja: { code: 'ja', name: 'Japanese', nativeName: '日本語', direction: 'ltr', voiceCode: 'ja-JP', region: 'Japan', popular: true },
  
  // Additional major languages
  ko: { code: 'ko', name: 'Korean', nativeName: '한국어', direction: 'ltr', voiceCode: 'ko-KR', region: 'Korea' },
  pt: { code: 'pt', name: 'Portuguese', nativeName: 'Português', direction: 'ltr', voiceCode: 'pt-BR', region: 'Brazil' },
  ru: { code: 'ru', name: 'Russian', nativeName: 'Русский', direction: 'ltr', voiceCode: 'ru-RU', region: 'Russia' },
  it: { code: 'it', name: 'Italian', nativeName: 'Italiano', direction: 'ltr', voiceCode: 'it-IT', region: 'Italy' },
  tr: { code: 'tr', name: 'Turkish', nativeName: 'Türkçe', direction: 'ltr', voiceCode: 'tr-TR', region: 'Turkey' },
  nl: { code: 'nl', name: 'Dutch', nativeName: 'Nederlands', direction: 'ltr', voiceCode: 'nl-NL', region: 'Netherlands' },
  pl: { code: 'pl', name: 'Polish', nativeName: 'Polski', direction: 'ltr', voiceCode: 'pl-PL', region: 'Poland' },
  vi: { code: 'vi', name: 'Vietnamese', nativeName: 'Tiếng Việt', direction: 'ltr', voiceCode: 'vi-VN', region: 'Vietnam' },
  th: { code: 'th', name: 'Thai', nativeName: 'ไทย', direction: 'ltr', voiceCode: 'th-TH', region: 'Thailand' },
  id: { code: 'id', name: 'Indonesian', nativeName: 'Bahasa Indonesia', direction: 'ltr', voiceCode: 'id-ID', region: 'Indonesia' },
  ms: { code: 'ms', name: 'Malay', nativeName: 'Bahasa Melayu', direction: 'ltr', voiceCode: 'ms-MY', region: 'Malaysia' },
  fa: { code: 'fa', name: 'Persian', nativeName: 'فارسی', direction: 'rtl', voiceCode: 'fa-IR', region: 'Iran' },
  he: { code: 'he', name: 'Hebrew', nativeName: 'עברית', direction: 'rtl', voiceCode: 'he-IL', region: 'Israel' },
  ta: { code: 'ta', name: 'Tamil', nativeName: 'தமிழ்', direction: 'ltr', voiceCode: 'ta-IN', region: 'India' },
  te: { code: 'te', name: 'Telugu', nativeName: 'తెలుగు', direction: 'ltr', voiceCode: 'te-IN', region: 'India' },
  mr: { code: 'mr', name: 'Marathi', nativeName: 'मराठी', direction: 'ltr', voiceCode: 'mr-IN', region: 'India' },
  gu: { code: 'gu', name: 'Gujarati', nativeName: 'ગુજરાતી', direction: 'ltr', voiceCode: 'gu-IN', region: 'India' },
  kn: { code: 'kn', name: 'Kannada', nativeName: 'ಕನ್ನಡ', direction: 'ltr', voiceCode: 'kn-IN', region: 'India' },
  ml: { code: 'ml', name: 'Malayalam', nativeName: 'മലയാളം', direction: 'ltr', voiceCode: 'ml-IN', region: 'India' },
  pa: { code: 'pa', name: 'Punjabi', nativeName: 'ਪੰਜਾਬੀ', direction: 'ltr', voiceCode: 'pa-IN', region: 'India' },
  
  // European languages
  uk: { code: 'uk', name: 'Ukrainian', nativeName: 'Українська', direction: 'ltr', voiceCode: 'uk-UA', region: 'Ukraine' },
  ro: { code: 'ro', name: 'Romanian', nativeName: 'Română', direction: 'ltr', voiceCode: 'ro-RO', region: 'Romania' },
  cs: { code: 'cs', name: 'Czech', nativeName: 'Čeština', direction: 'ltr', voiceCode: 'cs-CZ', region: 'Czech Republic' },
  sv: { code: 'sv', name: 'Swedish', nativeName: 'Svenska', direction: 'ltr', voiceCode: 'sv-SE', region: 'Sweden' },
  el: { code: 'el', name: 'Greek', nativeName: 'Ελληνικά', direction: 'ltr', voiceCode: 'el-GR', region: 'Greece' },
  hu: { code: 'hu', name: 'Hungarian', nativeName: 'Magyar', direction: 'ltr', voiceCode: 'hu-HU', region: 'Hungary' },
  da: { code: 'da', name: 'Danish', nativeName: 'Dansk', direction: 'ltr', voiceCode: 'da-DK', region: 'Denmark' },
  fi: { code: 'fi', name: 'Finnish', nativeName: 'Suomi', direction: 'ltr', voiceCode: 'fi-FI', region: 'Finland' },
  no: { code: 'no', name: 'Norwegian', nativeName: 'Norsk', direction: 'ltr', voiceCode: 'no-NO', region: 'Norway' },
  
  // African languages
  sw: { code: 'sw', name: 'Swahili', nativeName: 'Kiswahili', direction: 'ltr', voiceCode: 'sw-KE', region: 'Kenya' },
  am: { code: 'am', name: 'Amharic', nativeName: 'አማርኛ', direction: 'ltr', voiceCode: 'am-ET', region: 'Ethiopia' },
  zu: { code: 'zu', name: 'Zulu', nativeName: 'isiZulu', direction: 'ltr', voiceCode: 'zu-ZA', region: 'South Africa' },
  
  // Southeast Asian
  my: { code: 'my', name: 'Burmese', nativeName: 'မြန်မာ', direction: 'ltr', voiceCode: 'my-MM', region: 'Myanmar' },
  km: { code: 'km', name: 'Khmer', nativeName: 'ខ្មែរ', direction: 'ltr', voiceCode: 'km-KH', region: 'Cambodia' },
  lo: { code: 'lo', name: 'Lao', nativeName: 'ລາວ', direction: 'ltr', voiceCode: 'lo-LA', region: 'Laos' },
  
  // Others
  bg: { code: 'bg', name: 'Bulgarian', nativeName: 'Български', direction: 'ltr', voiceCode: 'bg-BG', region: 'Bulgaria' },
  hr: { code: 'hr', name: 'Croatian', nativeName: 'Hrvatski', direction: 'ltr', voiceCode: 'hr-HR', region: 'Croatia' },
  sk: { code: 'sk', name: 'Slovak', nativeName: 'Slovenčina', direction: 'ltr', voiceCode: 'sk-SK', region: 'Slovakia' },
  sl: { code: 'sl', name: 'Slovenian', nativeName: 'Slovenščina', direction: 'ltr', voiceCode: 'sl-SI', region: 'Slovenia' },
  lt: { code: 'lt', name: 'Lithuanian', nativeName: 'Lietuvių', direction: 'ltr', voiceCode: 'lt-LT', region: 'Lithuania' },
  lv: { code: 'lv', name: 'Latvian', nativeName: 'Latviešu', direction: 'ltr', voiceCode: 'lv-LV', region: 'Latvia' },
  et: { code: 'et', name: 'Estonian', nativeName: 'Eesti', direction: 'ltr', voiceCode: 'et-EE', region: 'Estonia' },
};

export const getPopularLanguages = (): Language[] => {
  return Object.values(LANGUAGES).filter(lang => lang.popular);
};

export const getAllLanguages = (): Language[] => {
  return Object.values(LANGUAGES).sort((a, b) => {
    if (a.popular && !b.popular) return -1;
    if (!a.popular && b.popular) return 1;
    return a.name.localeCompare(b.name);
  });
};

export const getLanguage = (code: string): Language | undefined => {
  return LANGUAGES[code];
};

export const detectBrowserLanguage = (): string => {
  const browserLang = navigator.language || navigator.languages?.[0] || 'en';
  const langCode = browserLang.split('-')[0].toLowerCase();
  
  // Return if we support this language
  if (LANGUAGES[langCode]) {
    return langCode;
  }
  
  // Default to English
  return 'en';
};

export const isRTL = (langCode: string): boolean => {
  return LANGUAGES[langCode]?.direction === 'rtl';
};
