import { useState } from "react";
import { useNavigate } from "react-router-dom";
import { supabase } from "@/integrations/supabase/client";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { toast } from "sonner";
import { Loader2 } from "lucide-react";

const Auth = () => {
  const navigate = useNavigate();
  const [loading, setLoading] = useState(false);
  const [emailOrPhone, setEmailOrPhone] = useState("");
  const [otp, setOtp] = useState("");
  const [showOtpInput, setShowOtpInput] = useState(false);
  const [fullName, setFullName] = useState("");
  const [isSignup, setIsSignup] = useState(false);

  const handleSendOtp = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      // Validate input is not empty
      if (!emailOrPhone || emailOrPhone.trim() === "") {
        throw new Error("ইমেইল বা মোবাইল নম্বর প্রয়োজন | Email or phone number is required");
      }

      const trimmedInput = emailOrPhone.trim();
      const isPhone = !trimmedInput.includes("@");
      
      if (isPhone) {
        // Validate Bangladeshi phone format
        if (!/^01\d{9}$/.test(trimmedInput)) {
          throw new Error("অবৈধ মোবাইল নম্বর। ফর্ম্যাট: 01XXXXXXXXX | Invalid mobile number. Format: 01XXXXXXXXX");
        }
      } else {
        // Validate email format
        if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(trimmedInput)) {
          throw new Error("অবৈধ ইমেইল | Invalid email format");
        }
      }

      // Send OTP
      const otpOptions = isPhone 
        ? { phone: `+880${trimmedInput.substring(1)}` }
        : { email: trimmedInput };

      const { error } = await supabase.auth.signInWithOtp({
        ...otpOptions,
        options: {
          data: fullName && isSignup ? { full_name: fullName } : undefined,
        },
      });

      if (error) throw error;

      setShowOtpInput(true);
      toast.success("যাচাইকরণ কোড পাঠানো হয়েছে! | Verification code sent!");
    } catch (error: any) {
      toast.error(error.message || "কোড পাঠাতে ব্যর্থ | Failed to send code");
    } finally {
      setLoading(false);
    }
  };

  const handleVerifyOtp = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      const isPhone = !emailOrPhone.includes("@");
      
      let error;
      if (isPhone) {
        const result = await supabase.auth.verifyOtp({
          phone: `+880${emailOrPhone.substring(1)}`,
          token: otp,
          type: 'sms',
        });
        error = result.error;
      } else {
        const result = await supabase.auth.verifyOtp({
          email: emailOrPhone,
          token: otp,
          type: 'email',
        });
        error = result.error;
      }

      if (error) throw error;

      toast.success("স্বাগতম! | Welcome!");
      navigate("/");
    } catch (error: any) {
      toast.error(error.message || "যাচাইকরণ ব্যর্থ | Verification failed");
    } finally {
      setLoading(false);
    }
  };

  const handleReset = () => {
    setShowOtpInput(false);
    setOtp("");
    setEmailOrPhone("");
    setFullName("");
    setIsSignup(false);
  };

  return (
    <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-background via-background to-primary/5 p-4">
      <Card className="w-full max-w-md">
        <CardHeader className="text-center">
          <div className="text-4xl mb-2">🤖</div>
          <CardTitle className="text-2xl">Smart Friend</CardTitle>
          <CardDescription>
            পাসওয়ার্ড ছাড়াই লগইন করুন | Login without password
          </CardDescription>
        </CardHeader>
        <CardContent>
          {!showOtpInput ? (
            <form onSubmit={handleSendOtp} className="space-y-4">
              {isSignup && (
                <div className="space-y-2">
                  <Label htmlFor="fullname">পূর্ণ নাম | Full Name</Label>
                  <Input
                    id="fullname"
                    type="text"
                    placeholder="আপনার নাম | Your Name"
                    value={fullName}
                    onChange={(e) => setFullName(e.target.value)}
                    required
                  />
                </div>
              )}
              <div className="space-y-2">
                <Label htmlFor="email-phone">
                  {isSignup ? "ইমেইল বা মোবাইল | Email or Mobile" : "ইমেইল বা মোবাইল | Email or Mobile"}
                </Label>
                <Input
                  id="email-phone"
                  type="text"
                  placeholder="your@email.com or 01XXXXXXXXX"
                  value={emailOrPhone}
                  onChange={(e) => setEmailOrPhone(e.target.value)}
                  required
                />
              </div>
              <Button type="submit" className="w-full" disabled={loading}>
                {loading && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
                কোড পাঠান | Send Code
              </Button>
              <Button
                type="button"
                variant="link"
                className="w-full text-sm"
                onClick={() => setIsSignup(!isSignup)}
              >
                {isSignup 
                  ? "একাউন্ট আছে? লগইন করুন | Have account? Login" 
                  : "নতুন একাউন্ট? সাইনআপ করুন | New account? Sign up"}
              </Button>
            </form>
          ) : (
            <form onSubmit={handleVerifyOtp} className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="otp">যাচাইকরণ কোড | Verification Code</Label>
                <Input
                  id="otp"
                  type="text"
                  placeholder="000000"
                  value={otp}
                  onChange={(e) => setOtp(e.target.value)}
                  required
                  maxLength={6}
                  className="text-center text-2xl tracking-widest"
                />
                <p className="text-xs text-muted-foreground text-center">
                  {emailOrPhone.includes("@") 
                    ? "আপনার ইমেইল চেক করুন | Check your email"
                    : "আপনার মোবাইলে কোড পাঠানো হয়েছে | Code sent to your mobile"}
                </p>
              </div>
              <Button type="submit" className="w-full" disabled={loading}>
                {loading && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
                যাচাই করুন | Verify
              </Button>
              <Button
                type="button"
                variant="outline"
                className="w-full"
                onClick={handleReset}
                disabled={loading}
              >
                পুনরায় শুরু করুন | Start Over
              </Button>
            </form>
          )}
        </CardContent>
      </Card>
    </div>
  );
};

export default Auth;