import { useState, useEffect } from "react";
import { OwnerGuard } from "@/components/OwnerGuard";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { brainEngine } from "@/core/brain/brain.engine";
import { Brain, CheckCircle, XCircle, Clock, Activity } from "lucide-react";
import { ScrollArea } from "@/components/ui/scroll-area";

const BrainConsole = () => {
  const [tasks, setTasks] = useState(brainEngine.getTasks());
  const [operations, setOperations] = useState(brainEngine.getOperations());
  const [memory, setMemory] = useState(brainEngine.getMemory());

  useEffect(() => {
    const interval = setInterval(() => {
      setTasks(brainEngine.getTasks());
      setOperations(brainEngine.getOperations());
      setMemory(brainEngine.getMemory());
    }, 2000);

    return () => clearInterval(interval);
  }, []);

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "completed":
        return <CheckCircle className="h-4 w-4 text-green-500" />;
      case "failed":
        return <XCircle className="h-4 w-4 text-red-500" />;
      case "running":
        return <Activity className="h-4 w-4 text-blue-500 animate-pulse" />;
      default:
        return <Clock className="h-4 w-4 text-muted-foreground" />;
    }
  };

  return (
    <OwnerGuard>
      <div className="min-h-screen bg-background p-6">
        <div className="max-w-7xl mx-auto space-y-6">
          <div className="flex items-center gap-3">
            <Brain className="h-8 w-8 text-primary" />
            <div>
              <h1 className="text-3xl font-bold text-foreground">AI Brain Console</h1>
              <p className="text-muted-foreground">System Intelligence & Operations</p>
            </div>
          </div>

          <div className="grid gap-6 md:grid-cols-2">
            {/* Active Tasks */}
            <Card>
              <CardHeader>
                <CardTitle>Active Tasks</CardTitle>
              </CardHeader>
              <CardContent>
                <ScrollArea className="h-[300px]">
                  {tasks.length === 0 ? (
                    <p className="text-muted-foreground text-sm">No active tasks</p>
                  ) : (
                    <div className="space-y-3">
                      {tasks.map((task) => (
                        <div
                          key={task.id}
                          className="p-3 border border-border rounded-lg space-y-2"
                        >
                          <div className="flex items-center justify-between">
                            <span className="font-medium text-sm">{task.description}</span>
                            {getStatusIcon(task.status)}
                          </div>
                          <Badge variant="outline">{task.status}</Badge>
                          <div className="text-xs text-muted-foreground space-y-1">
                            {task.subtasks.map((subtask, idx) => (
                              <div key={idx}>• {subtask}</div>
                            ))}
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </ScrollArea>
              </CardContent>
            </Card>

            {/* Recent Operations */}
            <Card>
              <CardHeader>
                <CardTitle>Recent Operations</CardTitle>
              </CardHeader>
              <CardContent>
                <ScrollArea className="h-[300px]">
                  {operations.length === 0 ? (
                    <p className="text-muted-foreground text-sm">No operations logged</p>
                  ) : (
                    <div className="space-y-2">
                      {operations.map((op) => (
                        <div
                          key={op.id}
                          className="p-2 border border-border rounded text-xs"
                        >
                          <div className="flex items-center justify-between mb-1">
                            <Badge variant="outline" className="text-xs">
                              {op.type}
                            </Badge>
                            {op.success ? (
                              <CheckCircle className="h-3 w-3 text-green-500" />
                            ) : (
                              <XCircle className="h-3 w-3 text-red-500" />
                            )}
                          </div>
                          <p className="text-muted-foreground">{op.details}</p>
                          <p className="text-muted-foreground text-[10px] mt-1">
                            {new Date(op.timestamp).toLocaleString()}
                          </p>
                        </div>
                      ))}
                    </div>
                  )}
                </ScrollArea>
              </CardContent>
            </Card>

            {/* Memory Snapshot */}
            <Card className="md:col-span-2">
              <CardHeader>
                <CardTitle>Memory Snapshot</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="grid gap-4 md:grid-cols-2">
                  <div>
                    <h3 className="font-semibold mb-2 text-sm">Short-Term Memory</h3>
                    <ScrollArea className="h-[200px]">
                      <div className="space-y-2">
                        {memory.shortTerm.slice(-10).map((item, idx) => (
                          <div key={idx} className="p-2 bg-muted rounded text-xs">
                            <Badge variant="secondary" className="text-[10px] mb-1">
                              {item.type}
                            </Badge>
                            <p className="text-muted-foreground line-clamp-2">
                              {item.content}
                            </p>
                          </div>
                        ))}
                      </div>
                    </ScrollArea>
                  </div>
                  <div>
                    <h3 className="font-semibold mb-2 text-sm">Long-Term Memory</h3>
                    <ScrollArea className="h-[200px]">
                      {memory.longTerm.length === 0 ? (
                        <p className="text-muted-foreground text-xs">
                          No long-term memories stored
                        </p>
                      ) : (
                        <div className="space-y-2">
                          {memory.longTerm.map((item) => (
                            <div key={item.id} className="p-2 bg-muted rounded text-xs">
                              <p className="text-muted-foreground">{item.summary}</p>
                              <Badge variant="outline" className="text-[10px] mt-1">
                                Importance: {item.importance}
                              </Badge>
                            </div>
                          ))}
                        </div>
                      )}
                    </ScrollArea>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </OwnerGuard>
  );
};

export default BrainConsole;
