import { useState } from "react";
import { useGlobalBrain } from "@/hooks/useGlobalBrain";
import { useOwnerCheck } from "@/hooks/useOwnerCheck";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Brain, Send, Loader2, Languages, Sparkles, Target } from "lucide-react";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";

interface Message {
  role: "user" | "assistant";
  content: string;
  suggestedModule?: string | null;
  emotion?: string;
  confidence?: number;
  timestamp: Date;
}

const BrainDemo = () => {
  const [messages, setMessages] = useState<Message[]>([]);
  const [input, setInput] = useState("");
  const [isBangla, setIsBangla] = useState(false);
  const { sendMessage, isProcessing, analyzeIntent } = useGlobalBrain();
  const { isOwner } = useOwnerCheck();

  const handleSend = async () => {
    if (!input.trim() || isProcessing) return;

    const userMessage: Message = {
      role: "user",
      content: input,
      timestamp: new Date(),
    };

    setMessages((prev) => [...prev, userMessage]);
    const userInput = input;
    setInput("");

    // Send to Global Brain
    const response = await sendMessage(userInput, isBangla ? "bn" : "en");

    if (response) {
      const assistantMessage: Message = {
        role: "assistant",
        content: response.replyText,
        suggestedModule: response.suggestedModule,
        emotion: response.emotion,
        confidence: response.confidence,
        timestamp: new Date(),
      };

      setMessages((prev) => [...prev, assistantMessage]);
    }
  };

  const getEmotionColor = (emotion?: string) => {
    switch (emotion) {
      case "confident":
        return "bg-green-500";
      case "friendly":
        return "bg-blue-500";
      case "caring":
        return "bg-purple-500";
      case "apologetic":
        return "bg-yellow-500";
      default:
        return "bg-gray-500";
    }
  };

  const getModuleColor = (module: string) => {
    const colors: Record<string, string> = {
      "smart-academy": "bg-blue-500",
      "smart-doctor": "bg-green-500",
      "smart-design-hub": "bg-purple-500",
      "smart-marketer": "bg-orange-500",
      "smart-bazar": "bg-pink-500",
      "smart-job": "bg-indigo-500",
      "smart-lawyer": "bg-red-500",
      "smart-news": "bg-cyan-500",
    };
    return colors[module] || "bg-gray-500";
  };

  return (
    <div className="h-screen flex flex-col bg-background">
      {/* Header */}
      <div className="border-b border-border bg-card">
        <div className="container mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-3">
              <Brain className="h-8 w-8 text-primary" />
              <div>
                <h1 className="text-2xl font-bold text-foreground">Global AI Brain Demo</h1>
                <p className="text-sm text-muted-foreground">
                  Multi-language, identity-aware, module-routing AI
                </p>
              </div>
            </div>

            <div className="flex items-center gap-4">
              <div className="flex items-center gap-2">
                <Switch
                  checked={isBangla}
                  onCheckedChange={setIsBangla}
                  id="language"
                />
                <Label htmlFor="language" className="cursor-pointer">
                  <Languages className="h-4 w-4 inline mr-1" />
                  {isBangla ? "বাংলা" : "English"}
                </Label>
              </div>
              {isOwner && (
                <Badge variant="outline" className="gap-1">
                  <Sparkles className="h-3 w-3" />
                  Owner Mode
                </Badge>
              )}
            </div>
          </div>
        </div>
      </div>

      {/* Chat Area */}
      <div className="flex-1 overflow-hidden">
        <div className="container mx-auto h-full p-4 max-w-4xl">
          <Card className="h-full flex flex-col">
            <CardHeader className="pb-3">
              <CardTitle>Test the Global Brain</CardTitle>
              <CardDescription>
                Try asking identity questions, requesting modules, or chatting in Bangla/English
              </CardDescription>
            </CardHeader>

            <CardContent className="flex-1 flex flex-col gap-4 overflow-hidden">
              <ScrollArea className="flex-1 pr-4">
                <div className="space-y-4">
                  {messages.length === 0 && (
                    <div className="text-center py-12 text-muted-foreground">
                      <Brain className="h-16 w-16 mx-auto mb-4 opacity-50" />
                      <p className="text-lg font-medium">Start a conversation</p>
                      <p className="text-sm mt-2">
                        Try: "Who created you?" or "আমি গণিত শিখতে চাই"
                      </p>
                    </div>
                  )}

                  {messages.map((message, index) => (
                    <div
                      key={index}
                      className={`flex ${
                        message.role === "user" ? "justify-end" : "justify-start"
                      }`}
                    >
                      <div
                        className={`max-w-[80%] rounded-lg p-4 ${
                          message.role === "user"
                            ? "bg-primary text-primary-foreground"
                            : "bg-muted"
                        }`}
                      >
                        <p className="whitespace-pre-wrap">{message.content}</p>

                        {message.role === "assistant" && (
                          <div className="flex gap-2 mt-3 flex-wrap">
                            {message.emotion && (
                              <Badge
                                variant="outline"
                                className={`${getEmotionColor(message.emotion)} text-white border-none`}
                              >
                                {message.emotion}
                              </Badge>
                            )}

                            {message.confidence !== undefined && (
                              <Badge variant="secondary">
                                <Target className="h-3 w-3 mr-1" />
                                {Math.round(message.confidence * 100)}% confident
                              </Badge>
                            )}

                            {message.suggestedModule && (
                              <Badge
                                className={`${getModuleColor(message.suggestedModule)} text-white`}
                              >
                                <Sparkles className="h-3 w-3 mr-1" />
                                {message.suggestedModule}
                              </Badge>
                            )}
                          </div>
                        )}
                      </div>
                    </div>
                  ))}

                  {isProcessing && (
                    <div className="flex justify-start">
                      <div className="bg-muted rounded-lg p-4">
                        <div className="flex items-center gap-2 text-muted-foreground">
                          <Loader2 className="h-4 w-4 animate-spin" />
                          <span>AI Brain processing...</span>
                        </div>
                      </div>
                    </div>
                  )}
                </div>
              </ScrollArea>

              <div className="flex gap-2">
                <Input
                  value={input}
                  onChange={(e) => setInput(e.target.value)}
                  onKeyDown={(e) => {
                    if (e.key === "Enter" && !e.shiftKey) {
                      e.preventDefault();
                      handleSend();
                    }
                  }}
                  placeholder={
                    isBangla
                      ? "আপনার বার্তা লিখুন..."
                      : "Type your message..."
                  }
                  disabled={isProcessing}
                />
                <Button onClick={handleSend} disabled={isProcessing || !input.trim()}>
                  {isProcessing ? (
                    <Loader2 className="h-4 w-4 animate-spin" />
                  ) : (
                    <Send className="h-4 w-4" />
                  )}
                </Button>
              </div>

              <div className="text-xs text-muted-foreground">
                <p className="font-medium mb-1">Example queries:</p>
                <ul className="space-y-1">
                  <li>• "Who created you?" / "তুমি কে তৈরি করেছে?"</li>
                  <li>• "I need help with math homework"</li>
                  <li>• "আমার একটা ডাক্তারের অ্যাপয়েন্টমেন্ট দরকার"</li>
                  <li>• "Create a poster design"</li>
                </ul>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
};

export default BrainDemo;
