import { useEffect, useState } from "react";
import { supabase } from "@/integrations/supabase/client";
import { CallHistoryItem } from "@/components/call/CallHistoryItem";
import { Card } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { ScrollArea } from "@/components/ui/scroll-area";
import { toast } from "sonner";
import { useNavigate } from "react-router-dom";
import { Loader2 } from "lucide-react";

type Call = {
  id: string;
  caller_id: string;
  callee_id: string;
  status: string;
  call_type: "voice" | "video";
  started_at: string;
  ended_at: string | null;
  created_at: string;
};

type ContactMap = {
  [key: string]: string;
};

const CallHistory = () => {
  const [calls, setCalls] = useState<Call[]>([]);
  const [loading, setLoading] = useState(true);
  const [currentUserId, setCurrentUserId] = useState<string>("");
  const [contacts, setContacts] = useState<ContactMap>({});
  const navigate = useNavigate();

  useEffect(() => {
    fetchCallHistory();
    setupRealtimeSubscription();
  }, []);

  const fetchCallHistory = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      setCurrentUserId(user.id);

      // Fetch calls
      const { data: callsData, error: callsError } = await supabase
        .from("calls")
        .select("*")
        .or(`caller_id.eq.${user.id},callee_id.eq.${user.id}`)
        .order("created_at", { ascending: false });

      if (callsError) throw callsError;

      setCalls((callsData || []) as Call[]);

      // Fetch contact names
      const contactIds = new Set<string>();
      callsData?.forEach((call) => {
        contactIds.add(call.caller_id);
        contactIds.add(call.callee_id);
      });
      contactIds.delete(user.id);

      const { data: profilesData, error: profilesError } = await supabase
        .from("profiles")
        .select("id, full_name")
        .in("id", Array.from(contactIds));

      if (profilesError) throw profilesError;

      const contactsMap: ContactMap = {};
      profilesData?.forEach((profile) => {
        contactsMap[profile.id] = profile.full_name;
      });

      setContacts(contactsMap);
    } catch (error) {
      console.error("Error fetching call history:", error);
      toast.error("Failed to load call history");
    } finally {
      setLoading(false);
    }
  };

  const setupRealtimeSubscription = async () => {
    const { data: { user } } = await supabase.auth.getUser();
    if (!user) return;

    const channel = supabase
      .channel("call-history-updates")
      .on(
        "postgres_changes",
        {
          event: "*",
          schema: "public",
          table: "calls",
          filter: `caller_id=eq.${user.id}`,
        },
        () => fetchCallHistory()
      )
      .on(
        "postgres_changes",
        {
          event: "*",
          schema: "public",
          table: "calls",
          filter: `callee_id=eq.${user.id}`,
        },
        () => fetchCallHistory()
      )
      .subscribe();

    return () => {
      supabase.removeChannel(channel);
    };
  };

  const handleCallBack = async (contactId: string, callType: "voice" | "video") => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) {
        toast.error("You must be logged in to make calls");
        return;
      }

      const { data, error } = await supabase
        .from("calls")
        .insert({
          caller_id: user.id,
          callee_id: contactId,
          call_type: callType,
          status: "ringing",
        })
        .select()
        .single();

      if (error) throw error;

      navigate(`/chat?call=${data.id}&initiator=true`);
    } catch (error) {
      console.error("Error initiating call:", error);
      toast.error("Failed to initiate call");
    }
  };

  const filterCalls = (status?: string) => {
    if (!status) return calls;
    return calls.filter((call) => call.status === status);
  };

  const missedCalls = calls.filter(
    (call) => call.status === "missed" && call.callee_id === currentUserId
  );

  if (loading) {
    return (
      <div className="flex items-center justify-center h-full">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }

  return (
    <div className="h-full flex flex-col">
      <div className="bg-gradient-to-r from-primary/10 to-primary/5 p-4 border-b">
        <h1 className="text-2xl font-bold">Call History</h1>
        <p className="text-sm text-muted-foreground">
          View your call history and duration
        </p>
      </div>

      <Tabs defaultValue="all" className="flex-1 flex flex-col">
        <TabsList className="w-full justify-start rounded-none border-b px-4">
          <TabsTrigger value="all">All ({calls.length})</TabsTrigger>
          <TabsTrigger value="missed">
            Missed ({missedCalls.length})
          </TabsTrigger>
          <TabsTrigger value="incoming">Incoming</TabsTrigger>
          <TabsTrigger value="outgoing">Outgoing</TabsTrigger>
        </TabsList>

        <ScrollArea className="flex-1">
          <div className="p-4">
            <TabsContent value="all" className="mt-0">
              {calls.length === 0 ? (
                <Card className="p-8 text-center">
                  <p className="text-muted-foreground">No call history yet</p>
                </Card>
              ) : (
                <div className="space-y-2">
                  {calls.map((call) => (
                    <CallHistoryItem
                      key={call.id}
                      call={call}
                      currentUserId={currentUserId}
                      contactName={
                        contacts[
                          call.caller_id === currentUserId
                            ? call.callee_id
                            : call.caller_id
                        ] || "Unknown"
                      }
                      onCallBack={handleCallBack}
                    />
                  ))}
                </div>
              )}
            </TabsContent>

            <TabsContent value="missed" className="mt-0">
              {missedCalls.length === 0 ? (
                <Card className="p-8 text-center">
                  <p className="text-muted-foreground">No missed calls</p>
                </Card>
              ) : (
                <div className="space-y-2">
                  {missedCalls.map((call) => (
                    <CallHistoryItem
                      key={call.id}
                      call={call}
                      currentUserId={currentUserId}
                      contactName={contacts[call.caller_id] || "Unknown"}
                      onCallBack={handleCallBack}
                    />
                  ))}
                </div>
              )}
            </TabsContent>

            <TabsContent value="incoming" className="mt-0">
              {filterCalls().filter((c) => c.callee_id === currentUserId).length === 0 ? (
                <Card className="p-8 text-center">
                  <p className="text-muted-foreground">No incoming calls</p>
                </Card>
              ) : (
                <div className="space-y-2">
                  {calls
                    .filter((c) => c.callee_id === currentUserId)
                    .map((call) => (
                      <CallHistoryItem
                        key={call.id}
                        call={call}
                        currentUserId={currentUserId}
                        contactName={contacts[call.caller_id] || "Unknown"}
                        onCallBack={handleCallBack}
                      />
                    ))}
                </div>
              )}
            </TabsContent>

            <TabsContent value="outgoing" className="mt-0">
              {filterCalls().filter((c) => c.caller_id === currentUserId).length === 0 ? (
                <Card className="p-8 text-center">
                  <p className="text-muted-foreground">No outgoing calls</p>
                </Card>
              ) : (
                <div className="space-y-2">
                  {calls
                    .filter((c) => c.caller_id === currentUserId)
                    .map((call) => (
                      <CallHistoryItem
                        key={call.id}
                        call={call}
                        currentUserId={currentUserId}
                        contactName={contacts[call.callee_id] || "Unknown"}
                        onCallBack={handleCallBack}
                      />
                    ))}
                </div>
              )}
            </TabsContent>
          </div>
        </ScrollArea>
      </Tabs>
    </div>
  );
};

export default CallHistory;
