import { useState, useRef, useEffect } from "react";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Button } from "@/components/ui/button";
import { Languages } from "lucide-react";
import { toast } from "sonner";
import { supabase } from "@/integrations/supabase/client";
import { MessageBubble } from "@/components/chat/MessageBubble";
import { TypingIndicator } from "@/components/chat/TypingIndicator";
import { ChatInput } from "@/components/chat/ChatInput";
import { VoiceInput } from "@/components/chat/VoiceInput";
import { CameraInput } from "@/components/chat/CameraInput";
import { FunctionExamples } from "@/components/chat/FunctionExamples";
import { WelcomeMessage } from "@/components/chat/WelcomeMessage";
import { useTextToSpeech } from "@/hooks/useTextToSpeech";
import { IncomingCallModal } from "@/components/call/IncomingCallModal";
import { VideoCallScreen } from "@/components/call/VideoCallScreen";
import { DateSeparator } from "@/components/chat/DateSeparator";
import { SmartFriendHeader } from "@/components/SmartFriendHeader";
import { OfflineBanner } from "@/components/chat/OfflineBanner";
import { AgentSelector, AgentMode, getSystemPrompt } from "@/components/chat/AgentSelector";
import { VoiceControls } from "@/components/chat/VoiceControls";
import { useSearchParams } from "react-router-dom";
import { isSameDay } from "date-fns";
import { useOfflineMode } from "@/hooks/useOfflineMode";
import { useOfflineIdentity } from "@/hooks/useOfflineIdentity";
import { useSemanticCache } from "@/hooks/useSemanticCache";
import { addRecentAction } from "@/components/home/RecentActions";
import { useWebSearch } from "@/hooks/useWebSearch";
import { useHybridBrain } from "@/hooks/useHybridBrain";
import { useKnowledgeCache } from "@/hooks/useKnowledgeCache";
import { useContextRegeneration } from "@/hooks/useContextRegeneration";
import { HybridBrainIndicator } from "@/components/chat/HybridBrainIndicator";
import { useIdentityEngine } from "@/hooks/useIdentityEngine";
import { useSafetyFilter } from "@/hooks/useSafetyFilter";
import { RoleSelector } from "@/components/chat/RoleSelector";
import { IdentityBadge } from "@/components/chat/IdentityBadge";

type Message = {
  role: "user" | "assistant";
  content: string;
  timestamp: Date;
};

const Chat = () => {
  const [messages, setMessages] = useState<Message[]>([]);
  const [input, setInput] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const [isBangla, setIsBangla] = useState(true);
  const [voiceEnabled, setVoiceEnabled] = useState(false);
  const [agentMode, setAgentMode] = useState<AgentMode>("general");
  const [recordingTime, setRecordingTime] = useState(0);
  const scrollRef = useRef<HTMLDivElement>(null);
  const { isSpeaking, speak, stop } = useTextToSpeech();
  const [searchParams] = useSearchParams();
  
  // Load chat history from localStorage
  useEffect(() => {
    const stored = localStorage.getItem("chat_history");
    if (stored) {
      const parsed = JSON.parse(stored);
      setMessages(
        parsed
          .map((m: any) => ({ ...m, timestamp: new Date(m.timestamp) }))
          .slice(-10) // Keep last 10 messages
      );
    }
  }, []);
  
  // Save messages to localStorage
  useEffect(() => {
    if (messages.length > 0) {
      localStorage.setItem("chat_history", JSON.stringify(messages.slice(-10)));
    }
  }, [messages]);
  
  // Offline mode hooks
  const { 
    isOnline, 
    queuedMessages, 
    cacheResponse, 
    findCachedAnswer, 
    queueMessage,
    getFallbackResponse 
  } = useOfflineMode();
  const { getOfflineResponse } = useOfflineIdentity();
  const { findCachedAnswer: findSemanticCache, cacheResponse: cacheSemanticResponse } = useSemanticCache();
  const { search: webSearch, searching: webSearching } = useWebSearch();
  
  // Hybrid Brain System
  const {
    query: hybridQuery,
    mode: brainMode,
    setMode: setBrainMode,
    status: brainStatus,
    failureCount,
    isOnline: hybridOnline,
    createMemorySnapshot,
    regenerateContext,
  } = useHybridBrain();
  
  const { searchCache, addToCache, prefillCache } = useKnowledgeCache();
  const { saveContext, regenerateFullContext } = useContextRegeneration();
  
  // Phase I: Identity & Safety System
  const {
    role,
    safetyLevel,
    identityContext,
    getSystemPrompt: getIdentitySystemPrompt,
    checkIdentityQuestion,
    switchRole,
    setLanguage: setIdentityLanguage,
  } = useIdentityEngine();
  
  const { 
    checkMessageSafety, 
    filterResponse,
    getSafetyGuidelines 
  } = useSafetyFilter();
  
  // Initialize knowledge cache on mount
  useEffect(() => {
    const initialized = localStorage.getItem("knowledge_cache_initialized");
    if (!initialized) {
      prefillCache(isBangla ? "bn" : "en");
      localStorage.setItem("knowledge_cache_initialized", "true");
    }
  }, [prefillCache, isBangla]);
  
  // Call management
  const [incomingCall, setIncomingCall] = useState<any>(null);
  const [activeCall, setActiveCall] = useState<string | null>(
    searchParams.get("call")
  );
  const [isCallInitiator, setIsCallInitiator] = useState(
    searchParams.get("initiator") === "true"
  );

  // Auto-scroll to bottom when messages change
  useEffect(() => {
    const scrollToBottom = () => {
      if (scrollRef.current) {
        scrollRef.current.scrollIntoView({ behavior: "smooth", block: "end" });
      }
    };
    
    // Immediate scroll
    scrollToBottom();
    
    // Delayed scroll to ensure content is rendered
    const timeout = setTimeout(scrollToBottom, 100);
    
    return () => clearTimeout(timeout);
  }, [messages, isLoading]);

  // Listen for incoming calls
  useEffect(() => {
    const setupCallListener = async () => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const callChannel = supabase
        .channel("incoming-calls")
        .on(
          "postgres_changes",
          {
            event: "INSERT",
            schema: "public",
            table: "calls",
            filter: `callee_id=eq.${user.id}`,
          },
          async (payload) => {
            const call = payload.new;
            if (call.status === "ringing") {
              // Fetch caller info
              const { data: callerProfile } = await supabase
                .from("profiles")
                .select("full_name")
                .eq("id", call.caller_id)
                .single();

              setIncomingCall({
                ...call,
                callerName: callerProfile?.full_name || "Unknown",
              });
            }
          }
        )
        .subscribe();

      return () => {
        supabase.removeChannel(callChannel);
      };
    };

    setupCallListener();
  }, []);

  const sendMessage = async (messageText: string) => {
    if (!messageText.trim() || isLoading) return;

    const language = isBangla ? "bn" : "en";

    // PHASE I: Safety check on user input
    const inputSafetyCheck = checkMessageSafety(messageText, role, safetyLevel);
    if (!inputSafetyCheck.isSafe && inputSafetyCheck.safeReplacement) {
      const safetyMessage: Message = {
        role: "assistant",
        content: inputSafetyCheck.safeReplacement,
        timestamp: new Date(),
      };
      setMessages((prev) => [...prev, safetyMessage]);
      if (voiceEnabled) speak(inputSafetyCheck.safeReplacement);
      return;
    }

    const userMessage: Message = { 
      role: "user", 
      content: messageText,
      timestamp: new Date()
    };
    setMessages((prev) => [...prev, userMessage]);
    setInput("");
    setIsLoading(true);
    
    // Add to recent actions
    addRecentAction({
      type: "chat",
      title: messageText.slice(0, 50),
      route: "/chat",
    });
    
    // Check if this is a web search request
    const searchTriggers = ['search web', 'web search', 'ওয়েবে খুঁজ', 'ওয়েব সার্চ', 'সার্চ কর', 'খোঁজ'];
    const isSearchRequest = searchTriggers.some(trigger => 
      messageText.toLowerCase().includes(trigger)
    );
    
    if (isSearchRequest) {
      try {
        const results = await webSearch({ query: messageText, maxResults: 3 });
        
        if (results.length > 0) {
          const summary = language === "bn" 
            ? `আমি ওয়েব সার্চ করেছি এবং ${results.length}টি ফলাফল পেয়েছি:\n\n` +
              results.map((r, i) => `${i + 1}. ${r.title}\n${r.snippet}\n🔗 ${r.url}`).join('\n\n')
            : `I found ${results.length} results from web search:\n\n` +
              results.map((r, i) => `${i + 1}. ${r.title}\n${r.snippet}\n🔗 ${r.url}`).join('\n\n');
          
          const assistantMessage: Message = {
            role: "assistant",
            content: summary,
            timestamp: new Date(),
          };
          setMessages((prev) => [...prev, assistantMessage]);
          if (voiceEnabled) speak(summary);
        } else {
          const noResultsMsg = language === "bn" 
            ? "দুঃখিত, কোনো ফলাফল পাওয়া যায়নি।"
            : "Sorry, no results found.";
          
          const assistantMessage: Message = {
            role: "assistant",
            content: noResultsMsg,
            timestamp: new Date(),
          };
          setMessages((prev) => [...prev, assistantMessage]);
          if (voiceEnabled) speak(noResultsMsg);
        }
      } catch (error) {
        console.error("Web search error:", error);
        const errorMsg = language === "bn"
          ? "দুঃখিত, ওয়েব সার্চে সমস্যা হয়েছে।"
          : "Sorry, there was an error with web search.";
        
        const assistantMessage: Message = {
          role: "assistant",
          content: errorMsg,
          timestamp: new Date(),
        };
        setMessages((prev) => [...prev, assistantMessage]);
      }
      setIsLoading(false);
      return;
    }
    
    let assistantContent = "";
    let toolCalls: any[] = [];

    try {
      // PHASE I: Check identity questions FIRST (highest priority)
      const identityAnswer = checkIdentityQuestion(messageText, language);
      if (identityAnswer) {
        const assistantMessage: Message = {
          role: "assistant",
          content: identityAnswer,
          timestamp: new Date(),
        };
        setMessages((prev) => [...prev, assistantMessage]);
        if (voiceEnabled) speak(identityAnswer);
        
        // Save to context
        saveContext([...messages, userMessage, assistantMessage]);
        createMemorySnapshot(messageText);
        
        setIsLoading(false);
        return;
      }

      // PHASE A: Check offline identity/safety answers SECOND
      const offlineAnswer = getOfflineResponse(messageText, language);
      if (offlineAnswer) {
        const assistantMessage: Message = {
          role: "assistant",
          content: offlineAnswer,
          timestamp: new Date(),
        };
        setMessages((prev) => [...prev, assistantMessage]);
        if (voiceEnabled) speak(offlineAnswer);
        
        // Save to context
        saveContext([...messages, userMessage, assistantMessage]);
        createMemorySnapshot(messageText);
        
        setIsLoading(false);
        return;
      }

      // PHASE H: Use Hybrid Brain System
      // Check knowledge cache first
      const cachedKnowledge = searchCache(messageText, language);
      if (cachedKnowledge) {
        const assistantMessage: Message = {
          role: "assistant",
          content: `[📚 ${language === "bn" ? "জ্ঞান ভান্ডার" : "Knowledge Cache"}] ${cachedKnowledge.answer}`,
          timestamp: new Date(),
        };
        setMessages((prev) => [...prev, assistantMessage]);
        if (voiceEnabled) speak(cachedKnowledge.answer);
        
        saveContext([...messages, userMessage, assistantMessage]);
        createMemorySnapshot(messageText);
        
        setIsLoading(false);
        return;
      }

      // Check semantic cache
      const semanticCached = await findSemanticCache(messageText, language);
      if (semanticCached) {
        const cachedMessage: Message = {
          role: "assistant",
          content: semanticCached,
          timestamp: new Date(),
        };
        setMessages((prev) => [...prev, cachedMessage]);
        if (voiceEnabled) speak(semanticCached);
        
        saveContext([...messages, userMessage, cachedMessage]);
        createMemorySnapshot(messageText);
        
        setIsLoading(false);
        return;
      }

      // Use Hybrid Brain query
      const hybridResponse = await hybridQuery(
        messageText,
        [...messages, userMessage],
        language
      );

      const sourceIndicator = 
        hybridResponse.source === "online" ? "🌐" :
        hybridResponse.source === "offline" ? "📱" : "💾";

      // PHASE I: Apply safety filter to response
      const rawResponse = `${sourceIndicator} ${hybridResponse.answer}`;
      const filteredContent = filterResponse(rawResponse, role, safetyLevel, language);

      assistantContent = filteredContent;

      const assistantMessage: Message = {
        role: "assistant",
        content: assistantContent,
        timestamp: new Date(),
      };
      setMessages((prev) => [...prev, assistantMessage]);

      // Cache the response
      cacheSemanticResponse(messageText, hybridResponse.answer, language);
      
      // Add to knowledge cache if high confidence
      if (hybridResponse.confidence > 0.9 && hybridResponse.source === "online") {
        const category = detectCategory(messageText);
        addToCache({
          category,
          question: messageText,
          answer: hybridResponse.answer,
          language,
        });
      }

      // Save context
      saveContext([...messages, userMessage, assistantMessage]);
      createMemorySnapshot(messageText);

      if (voiceEnabled) {
        speak(assistantContent);
      }
    } catch (error: any) {
      console.error("Error sending message:", error);
      
      // Fallback to offline on error
      const fallbackMsg = language === "bn"
        ? "দুঃখিত, একটি সমস্যা হয়েছে। অফলাইন মোডে স্যুইচ করা হচ্ছে..."
        : "Sorry, there was an error. Switching to offline mode...";
      
      toast.error(fallbackMsg);

      const assistantMessage: Message = {
        role: "assistant",
        content: getFallbackResponse(language),
        timestamp: new Date(),
      };
      setMessages((prev) => [...prev, assistantMessage]);
    } finally {
      setIsLoading(false);
    }
  };

  // Helper function to detect category
  const detectCategory = (text: string): any => {
    const lower = text.toLowerCase();
    if (lower.includes("গণিত") || lower.includes("math")) return "math";
    if (lower.includes("আইসিটি") || lower.includes("ict") || lower.includes("computer")) return "ict";
    if (lower.includes("বিজ্ঞান") || lower.includes("science")) return "science";
    if (lower.includes("স্বাস্থ্য") || lower.includes("health") || lower.includes("medical")) return "medical";
    if (lower.includes("ইতিহাস") || lower.includes("history")) return "history";
    return "general_knowledge";
  };

  const handleFileSelect = async (file: File) => {
    toast.info(isBangla ? "ফাইল আপলোড ফিচার শীঘ্রই আসছে" : "File upload feature coming soon");
  };

  const toggleLanguage = () => {
    const newLang = !isBangla;
    setIsBangla(newLang);
    setIdentityLanguage(newLang ? 'bn' : 'en');
    toast.success(isBangla ? "Language: English" : "ভাষা: বাংলা");
  };

  const toggleVoice = () => {
    if (isSpeaking) {
      stop();
    }
    setVoiceEnabled(!voiceEnabled);
  };

  const handleAcceptCall = async () => {
    if (!incomingCall) return;
    
    await supabase
      .from("calls")
      .update({ status: "active" })
      .eq("id", incomingCall.id);

    setActiveCall(incomingCall.id);
    setIsCallInitiator(false);
    setIncomingCall(null);
  };

  const handleDeclineCall = async () => {
    if (!incomingCall) return;

    await supabase
      .from("calls")
      .update({ status: "declined", ended_at: new Date().toISOString() })
      .eq("id", incomingCall.id);

    setIncomingCall(null);
  };

  const handleEndCall = () => {
    setActiveCall(null);
    setIsCallInitiator(false);
  };

  if (activeCall) {
    return (
      <VideoCallScreen
        callId={activeCall}
        isInitiator={isCallInitiator}
        onEndCall={handleEndCall}
        callerName={incomingCall?.callerName}
      />
    );
  }

  return (
    <>
      <IncomingCallModal
        isOpen={!!incomingCall}
        callerName={incomingCall?.callerName || "Unknown"}
        callType={incomingCall?.call_type || "voice"}
        onAccept={handleAcceptCall}
        onDecline={handleDeclineCall}
      />

      <div className="flex flex-col h-full">
        {/* Header Section */}
        <div className="bg-gradient-to-r from-primary/10 to-primary/5 border-b backdrop-blur-sm">
          <div className="max-w-4xl mx-auto">
            <div className="flex items-center justify-between p-4">
              <SmartFriendHeader />
              <Button
                variant="outline"
                size="sm"
                onClick={toggleLanguage}
                className="gap-2"
              >
                <Languages className="h-4 w-4" />
                {isBangla ? "English" : "বাংলা"}
              </Button>
            </div>
            <OfflineBanner 
              isOnline={isOnline} 
              queuedCount={queuedMessages.length}
              language={isBangla ? "bn" : "en"}
            />
          </div>
        </div>

        {/* Agent Mode Selector */}
        <AgentSelector mode={agentMode} onModeChange={setAgentMode} />
        
        {/* Phase I: Role Selector & Identity Badge */}
        <div className="max-w-4xl mx-auto px-4 py-2 flex items-center justify-between gap-4">
          <RoleSelector role={role} onRoleChange={switchRole} />
          <IdentityBadge role={role} safetyLevel={safetyLevel} language={isBangla ? 'bn' : 'en'} />
        </div>
        
        {/* Voice Controls */}
        <VoiceControls
          voiceEnabled={voiceEnabled}
          isRecording={false}
          isSpeaking={isSpeaking}
          recordingTime={recordingTime}
          onToggleVoice={() => setVoiceEnabled(!voiceEnabled)}
          onToggleSpeaking={() => (isSpeaking ? stop() : null)}
        />

      <ScrollArea className="flex-1 p-4">
        <div className="space-y-3 max-w-4xl mx-auto pb-32">
          {messages.length === 0 && (
            <div className="space-y-6 animate-fade-in">
              <WelcomeMessage />
              <FunctionExamples onExampleClick={setInput} isBangla={isBangla} />
            </div>
          )}

          {messages.map((message, index) => {
            const showDateSeparator = 
              index === 0 || 
              !isSameDay(message.timestamp, messages[index - 1].timestamp);

            return (
              <div key={index}>
                {showDateSeparator && (
                  <DateSeparator date={message.timestamp} isBangla={isBangla} />
                )}
                <MessageBubble
                  role={message.role}
                  content={message.content}
                  timestamp={message.timestamp}
                  isBangla={isBangla}
                />
              </div>
            );
          })}

          {isLoading && <TypingIndicator isBangla={isBangla} />}
          <div ref={scrollRef} className="h-4" />
        </div>
      </ScrollArea>

      <div className="fixed bottom-16 left-0 right-0 bg-background border-t border-border p-4">
        <div className="max-w-4xl mx-auto flex items-center gap-2">
          <VoiceInput onTranscript={(text) => sendMessage(text)} />
          <CameraInput onCapture={(imageData) => console.log("Image captured:", imageData)} />
          <div className="flex-1">
            <ChatInput
              input={input}
              setInput={setInput}
              onSend={() => sendMessage(input)}
              isLoading={isLoading}
              isBangla={isBangla}
              isSpeaking={isSpeaking}
              onToggleVoice={toggleVoice}
              onFileSelect={handleFileSelect}
            />
          </div>
        </div>
      </div>
      </div>
    </>
  );
};

export default Chat;