import { useEffect, useState } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Phone, Video, User } from "lucide-react";
import { toast } from "sonner";
import { useNavigate } from "react-router-dom";

type Profile = {
  id: string;
  full_name: string;
  avatar_url: string | null;
  phone: string | null;
};

const Contacts = () => {
  const [contacts, setContacts] = useState<Profile[]>([]);
  const [loading, setLoading] = useState(true);
  const navigate = useNavigate();

  useEffect(() => {
    fetchContacts();
  }, []);

  const fetchContacts = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const { data, error } = await supabase
        .from("profiles")
        .select("id, full_name, avatar_url, phone")
        .neq("id", user.id);

      if (error) throw error;
      setContacts(data || []);
    } catch (error) {
      console.error("Error fetching contacts:", error);
      toast.error("Failed to load contacts");
    } finally {
      setLoading(false);
    }
  };

  const initiateCall = async (contactId: string, callType: "voice" | "video") => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) {
        toast.error("You must be logged in to make calls");
        return;
      }

      const { data, error } = await supabase
        .from("calls")
        .insert({
          caller_id: user.id,
          callee_id: contactId,
          call_type: callType,
          status: "ringing",
        })
        .select()
        .single();

      if (error) throw error;

      navigate(`/chat?call=${data.id}&initiator=true`);
    } catch (error) {
      console.error("Error initiating call:", error);
      toast.error("Failed to initiate call");
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-full">
        <p className="text-muted-foreground">Loading contacts...</p>
      </div>
    );
  }

  return (
    <div className="h-full p-4">
      <div className="max-w-4xl mx-auto">
        <h1 className="text-2xl font-bold mb-6">Contacts</h1>

        {contacts.length === 0 ? (
          <Card className="p-8 text-center">
            <p className="text-muted-foreground">No contacts found</p>
          </Card>
        ) : (
          <div className="grid gap-3">
            {contacts.map((contact) => (
              <Card key={contact.id} className="p-4">
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-3">
                    <div className="h-12 w-12 rounded-full bg-primary/10 flex items-center justify-center">
                      {contact.avatar_url ? (
                        <img
                          src={contact.avatar_url}
                          alt={contact.full_name}
                          className="h-12 w-12 rounded-full object-cover"
                        />
                      ) : (
                        <User className="h-6 w-6 text-primary" />
                      )}
                    </div>
                    <div>
                      <p className="font-medium">{contact.full_name}</p>
                      {contact.phone && (
                        <p className="text-sm text-muted-foreground">
                          {contact.phone}
                        </p>
                      )}
                    </div>
                  </div>

                  <div className="flex gap-2">
                    <Button
                      variant="outline"
                      size="icon"
                      onClick={() => initiateCall(contact.id, "voice")}
                    >
                      <Phone className="h-4 w-4" />
                    </Button>
                    <Button
                      variant="outline"
                      size="icon"
                      onClick={() => initiateCall(contact.id, "video")}
                    >
                      <Video className="h-4 w-4" />
                    </Button>
                  </div>
                </div>
              </Card>
            ))}
          </div>
        )}
      </div>
    </div>
  );
};

export default Contacts;
