import { useState, useEffect } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { toast } from "sonner";
import { supabase } from "@/integrations/supabase/client";
import type { Json } from "@/integrations/supabase/types";
import { Brain, Lock, Activity, Database, Clock, AlertCircle, FileText, Bug, Code2, Sparkles, CheckCircle2, Scan, Wand2, Package } from "lucide-react";
import OnlineUsers from "@/components/OnlineUsers";
import Dashboard from "@/core-brain/dashboard";
import CriticalNotifications from "@/core-brain/criticalNotifications";
import PatchHistory from "@/core-brain/patchHistory";
import PendingPatches from "@/core-brain/pendingPatches";
import GeneratedTests from "@/core-brain/generatedTests";
import { CoreBrainAnalytics } from "@/core-brain/analytics";

type Log = {
  id: string;
  type: string;
  message: string;
  details: Json | null;
  created_at: string;
  user_id?: string | null;
};

type Plan = {
  id: string;
  title: string;
  description: string;
  status: string;
  created_at: string;
  created_by: string;
  updated_at: string;
};

type ScanResult = {
  id: string;
  scan_type: string;
  file_path: string | null;
  issue_type: string;
  severity: string;
  message: string;
  details: Json | null;
  resolved: boolean;
  created_at: string;
};

type FixPlan = {
  id: string;
  title: string;
  description: string;
  priority: string;
  status: string;
  ai_suggestion: string | null;
  created_at: string;
  created_by: string;
  updated_at: string;
  scan_result_id: string | null;
};

type CodePatch = {
  id: string;
  fix_plan_id: string | null;
  file_path: string;
  original_code: string | null;
  patched_code: string;
  status: string;
  created_at: string;
  created_by: string;
  applied_at: string | null;
};

type ModuleScaffold = {
  moduleName: string;
  description: string;
  features: string[];
  routes: { path: string; component: string; code: string }[];
  components: { name: string; path: string; code: string }[];
  database: { schema: string; migrations: string }[];
  edgeFunctions: { name: string; code: string }[];
};

const CoreBrain = () => {
  const [upgradeIdea, setUpgradeIdea] = useState("");
  const [planTitle, setPlanTitle] = useState("");
  const [isAnalyzing, setIsAnalyzing] = useState(false);
  const [isScanning, setIsScanning] = useState(false);
  const [isGeneratingModule, setIsGeneratingModule] = useState(false);
  const [logs, setLogs] = useState<Log[]>([]);
  const [plans, setPlans] = useState<Plan[]>([]);
  const [scanResults, setScanResults] = useState<ScanResult[]>([]);
  const [fixPlans, setFixPlans] = useState<FixPlan[]>([]);
  const [codePatches, setCodePatches] = useState<CodePatch[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [moduleScaffold, setModuleScaffold] = useState<ModuleScaffold | null>(null);
  const [moduleName, setModuleName] = useState("");
  const [moduleDescription, setModuleDescription] = useState("");
  const [moduleFeatures, setModuleFeatures] = useState<string[]>([""]);

  const systemStatus = {
    apiOnline: true,
    databaseStatus: "healthy",
    lastDeploy: new Date().toISOString().split('T')[0] + " " + new Date().toLocaleTimeString('en-US', { hour12: false }),
    uptime: "99.9%"
  };

  useEffect(() => {
    fetchLogs();
    fetchPlans();
    fetchScanResults();
    fetchFixPlans();
    fetchCodePatches();
  }, []);

  // Real-time collaboration subscriptions
  useEffect(() => {
    const logsChannel = supabase
      .channel('core_brain_logs_changes')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'core_brain_logs'
        },
        (payload) => {
          if (payload.eventType === 'INSERT') {
            setLogs((prev) => [payload.new as Log, ...prev].slice(0, 20));
            toast.info(`New log: ${(payload.new as Log).message}`);
          }
        }
      )
      .subscribe();

    const plansChannel = supabase
      .channel('core_brain_plans_changes')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'core_brain_plans'
        },
        (payload) => {
          if (payload.eventType === 'INSERT') {
            setPlans((prev) => [payload.new as Plan, ...prev]);
            toast.success(`New plan created: ${(payload.new as Plan).title}`);
          } else if (payload.eventType === 'UPDATE') {
            setPlans((prev) =>
              prev.map((plan) =>
                plan.id === (payload.new as Plan).id ? (payload.new as Plan) : plan
              )
            );
            toast.info(`Plan updated: ${(payload.new as Plan).title}`);
          } else if (payload.eventType === 'DELETE') {
            setPlans((prev) => prev.filter((plan) => plan.id !== (payload.old as Plan).id));
          }
        }
      )
      .subscribe();

    const scanResultsChannel = supabase
      .channel('scan_results_changes')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'scan_results'
        },
        (payload) => {
          if (payload.eventType === 'INSERT') {
            setScanResults((prev) => [payload.new as ScanResult, ...prev].slice(0, 50));
            const result = payload.new as ScanResult;
            toast.warning(`New ${result.severity} issue detected: ${result.message}`);
          } else if (payload.eventType === 'UPDATE') {
            setScanResults((prev) =>
              prev.map((result) =>
                result.id === (payload.new as ScanResult).id ? (payload.new as ScanResult) : result
              )
            );
          }
        }
      )
      .subscribe();

    const fixPlansChannel = supabase
      .channel('fix_plans_changes')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'fix_plans'
        },
        (payload) => {
          if (payload.eventType === 'INSERT') {
            setFixPlans((prev) => [payload.new as FixPlan, ...prev].slice(0, 20));
            toast.success(`New fix plan: ${(payload.new as FixPlan).title}`);
          } else if (payload.eventType === 'UPDATE') {
            setFixPlans((prev) =>
              prev.map((plan) =>
                plan.id === (payload.new as FixPlan).id ? (payload.new as FixPlan) : plan
              )
            );
            toast.info(`Fix plan updated: ${(payload.new as FixPlan).title}`);
          } else if (payload.eventType === 'DELETE') {
            setFixPlans((prev) => prev.filter((plan) => plan.id !== (payload.old as FixPlan).id));
          }
        }
      )
      .subscribe();

    const codePatchesChannel = supabase
      .channel('code_patches_changes')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'code_patches'
        },
        (payload) => {
          if (payload.eventType === 'INSERT') {
            setCodePatches((prev) => [payload.new as CodePatch, ...prev].slice(0, 20));
            toast.success(`New code patch created for: ${(payload.new as CodePatch).file_path}`);
          } else if (payload.eventType === 'UPDATE') {
            setCodePatches((prev) =>
              prev.map((patch) =>
                patch.id === (payload.new as CodePatch).id ? (payload.new as CodePatch) : patch
              )
            );
            const updatedPatch = payload.new as CodePatch;
            if (updatedPatch.status === 'applied') {
              toast.success(`Code patch applied: ${updatedPatch.file_path}`);
            }
          } else if (payload.eventType === 'DELETE') {
            setCodePatches((prev) => prev.filter((patch) => patch.id !== (payload.old as CodePatch).id));
          }
        }
      )
      .subscribe();

    // Cleanup subscriptions on unmount
    return () => {
      supabase.removeChannel(logsChannel);
      supabase.removeChannel(plansChannel);
      supabase.removeChannel(scanResultsChannel);
      supabase.removeChannel(fixPlansChannel);
      supabase.removeChannel(codePatchesChannel);
    };
  }, []);

  const fetchLogs = async (): Promise<void> => {
    try {
      const { data, error } = await supabase
        .from("core_brain_logs")
        .select("*")
        .order("created_at", { ascending: false })
        .limit(20);

      if (error) throw error;
      setLogs(data || []);
    } catch (error) {
      console.error("Error fetching logs:", error);
      toast.error("Failed to load logs");
    } finally {
      setIsLoading(false);
    }
  };

  const fetchPlans = async (): Promise<void> => {
    try {
      const { data, error } = await supabase
        .from("core_brain_plans")
        .select("*")
        .order("created_at", { ascending: false });

      if (error) throw error;
      setPlans(data || []);
    } catch (error) {
      console.error("Error fetching plans:", error);
    }
  };

  const fetchScanResults = async (): Promise<void> => {
    try {
      const { data, error } = await supabase
        .from("scan_results")
        .select("*")
        .order("created_at", { ascending: false })
        .limit(50);

      if (error) throw error;
      setScanResults(data || []);
    } catch (error) {
      console.error("Error fetching scan results:", error);
    }
  };

  const fetchFixPlans = async (): Promise<void> => {
    try {
      const { data, error } = await supabase
        .from("fix_plans")
        .select("*")
        .order("created_at", { ascending: false })
        .limit(20);

      if (error) throw error;
      setFixPlans(data || []);
    } catch (error) {
      console.error("Error fetching fix plans:", error);
    }
  };

  const fetchCodePatches = async (): Promise<void> => {
    try {
      const { data, error } = await supabase
        .from("code_patches")
        .select("*")
        .order("created_at", { ascending: false })
        .limit(20);

      if (error) throw error;
      setCodePatches(data || []);
    } catch (error) {
      console.error("Error fetching code patches:", error);
    }
  };

  const handleSaveUpgrade = async (): Promise<void> => {
    if (!upgradeIdea.trim() || !planTitle.trim()) {
      toast.error("Please provide both title and description");
      return;
    }

    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) {
        toast.error("You must be logged in");
        return;
      }

      const { error } = await supabase
        .from("core_brain_plans")
        .insert({
          title: planTitle,
          description: upgradeIdea,
          created_by: user.id,
          status: "pending"
        });

      if (error) throw error;

      toast.success("Upgrade plan saved successfully");
      setPlanTitle("");
      setUpgradeIdea("");
      fetchPlans();
    } catch (error) {
      console.error("Error saving plan:", error);
      toast.error("Failed to save upgrade plan");
    }
  };

  const handleRunProjectScan = async (): Promise<void> => {
    setIsScanning(true);
    try {
      toast.info("Starting project scan...");
      
      const { data, error } = await supabase.functions.invoke('core-brain-scanner', {
        body: {
          scanType: 'full',
          options: {
            checkErrors: true,
            checkWarnings: true,
            checkUnused: true,
            checkPerformance: true
          }
        }
      });

      if (error) throw error;

      toast.success(`Scan complete: ${data?.results?.length || 0} issues found`);
      fetchScanResults();
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : "Failed to scan project";
      console.error("Error scanning project:", error);
      toast.error(errorMessage);
    } finally {
      setIsScanning(false);
    }
  };

  const handleAnalyzeErrors = async (): Promise<void> => {
    if (logs.length === 0) {
      toast.error("No logs to analyze");
      return;
    }

    setIsAnalyzing(true);
    try {
      const recentErrors = logs.filter(log => log.type === 'error').slice(0, 5);
      
      const { data, error } = await supabase.functions.invoke('core-brain-analyze', {
        body: {
          type: 'error_analysis',
          context: {
            logs: recentErrors.map(log => ({
              message: log.message,
              details: log.details,
              created_at: log.created_at
            }))
          }
        }
      });

      if (error) throw error;
      
      if (data?.result) {
        toast.success("Error analysis complete");
        fetchFixPlans();
      }
    } catch (error) {
      console.error("Error analyzing:", error);
      toast.error("Failed to analyze errors");
    } finally {
      setIsAnalyzing(false);
    }
  };

  const handleGenerateFixPlan = async (scanResult: ScanResult): Promise<void> => {
    try {
      toast.info("Generating fix plan...");
      
      const { data, error } = await supabase.functions.invoke('core-brain-analyze', {
        body: {
          type: 'fix_suggestion',
          context: {
            description: scanResult.message,
            details: scanResult.details,
            severity: scanResult.severity
          }
        }
      });

      if (error) throw error;

      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      let aiResponse: { title?: string; description?: string; priority?: string };
      try {
        aiResponse = JSON.parse(data.result as string);
      } catch {
        aiResponse = {
          title: `Fix: ${scanResult.message}`,
          description: data.result as string,
          priority: scanResult.severity
        };
      }

      const { error: insertError } = await supabase
        .from("fix_plans")
        .insert({
          scan_result_id: scanResult.id,
          title: aiResponse.title || `Fix: ${scanResult.message}`,
          description: aiResponse.description || (data.result as string),
          priority: aiResponse.priority || scanResult.severity,
          ai_suggestion: data.result as string,
          created_by: user.id,
          status: 'pending'
        });

      if (insertError) throw insertError;

      toast.success("Fix plan generated");
      fetchFixPlans();
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : "Failed to generate fix plan";
      console.error("Error generating fix plan:", error);
      toast.error(errorMessage);
    }
  };

  const handleGenerateCodePatch = async (fixPlan: FixPlan): Promise<void> => {
    try {
      toast.info("Generating code patch...");
      
      const { data, error } = await supabase.functions.invoke('core-brain-analyze', {
        body: {
          type: 'code_patch',
          context: {
            filePath: 'auto-generated',
            task: fixPlan.description,
            existing: ''
          }
        }
      });

      if (error) throw error;

      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      const { error: insertError } = await supabase
        .from("code_patches")
        .insert({
          fix_plan_id: fixPlan.id,
          file_path: 'auto-generated.ts',
          patched_code: data.result as string,
          created_by: user.id,
          status: 'pending'
        });

      if (insertError) throw insertError;

      toast.success("Code patch generated");
      fetchCodePatches();
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : "Failed to generate code patch";
      console.error("Error generating patch:", error);
      toast.error(errorMessage);
    }
  };

  const handleApproveFixPlan = async (id: string): Promise<void> => {
    try {
      const { error } = await supabase
        .from("fix_plans")
        .update({ status: 'approved' })
        .eq("id", id);

      if (error) throw error;
      toast.success("Fix plan approved");
      fetchFixPlans();
    } catch (error) {
      console.error("Error approving:", error);
      toast.error("Failed to approve fix plan");
    }
  };

  const handleRejectFixPlan = async (id: string): Promise<void> => {
    try {
      const { error } = await supabase
        .from("fix_plans")
        .update({ status: 'rejected' })
        .eq("id", id);

      if (error) throw error;
      toast.success("Fix plan rejected");
      fetchFixPlans();
    } catch (error) {
      console.error("Error rejecting:", error);
      toast.error("Failed to reject fix plan");
    }
  };

  const handleMarkResolved = async (id: string): Promise<void> => {
    try {
      const { error } = await supabase
        .from("scan_results")
        .update({ resolved: true })
        .eq("id", id);

      if (error) throw error;
      toast.success("Issue marked as resolved");
      fetchScanResults();
    } catch (error) {
      console.error("Error marking resolved:", error);
      toast.error("Failed to mark as resolved");
    }
  };

  const handleGenerateModule = async (): Promise<void> => {
    if (!moduleName.trim() || !moduleDescription.trim()) {
      toast.error("Please provide module name and description");
      return;
    }

    const validFeatures = moduleFeatures.filter(f => f.trim());
    if (validFeatures.length === 0) {
      toast.error("Please add at least one feature");
      return;
    }

    setIsGeneratingModule(true);
    try {
      toast.info("Generating module scaffold...");

      const { data, error } = await supabase.functions.invoke('core-brain-analyze', {
        body: {
          type: 'module_scaffold',
          context: {
            moduleName,
            description: moduleDescription,
            features: validFeatures
          }
        }
      });

      if (error) throw error;

      let scaffold: ModuleScaffold;
      try {
        scaffold = JSON.parse(data.result as string);
      } catch {
        toast.error("Failed to parse module scaffold");
        return;
      }

      setModuleScaffold(scaffold);
      toast.success("Module scaffold generated! Review below.");

      // Log the generation
      await supabase.from('core_brain_logs').insert({
        type: 'info',
        message: `Module scaffold generated: ${moduleName}`,
        details: { moduleName, features: validFeatures }
      });

    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : "Failed to generate module scaffold";
      console.error("Error generating module:", error);
      toast.error(errorMessage);
    } finally {
      setIsGeneratingModule(false);
    }
  };

  const addFeatureField = (): void => {
    setModuleFeatures([...moduleFeatures, ""]);
  };

  const updateFeature = (index: number, value: string): void => {
    const updated = [...moduleFeatures];
    updated[index] = value;
    setModuleFeatures(updated);
  };

  const removeFeature = (index: number): void => {
    if (moduleFeatures.length > 1) {
      setModuleFeatures(moduleFeatures.filter((_, i) => i !== index));
    }
  };

  const getSeverityColor = (type: string): "destructive" | "default" | "secondary" => {
    switch (type) {
      case "error": return "destructive";
      case "warning": return "default";
      case "info": return "secondary";
      default: return "default";
    }
  };

  const formatDate = (dateString: string): string => {
    const date = new Date(dateString);
    return date.toLocaleString('en-US', {
      year: 'numeric',
      month: '2-digit',
      day: '2-digit',
      hour: '2-digit',
      minute: '2-digit',
      second: '2-digit',
      hour12: false
    });
  };

  return (
    <div className="h-full overflow-auto pb-20">
      <div className="p-4 sm:p-6 max-w-7xl mx-auto space-y-6">
        {/* Header */}
        <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-4 mb-6">
          <div className="flex items-center gap-3">
            <div className="relative">
              <div className="absolute inset-0 bg-gradient-to-br from-blue-500 to-pink-500 rounded-lg blur-lg opacity-40" />
              <div className="relative bg-gradient-to-br from-blue-500 to-pink-500 p-2 rounded-lg">
                <Brain className="w-6 h-6 text-white" />
              </div>
            </div>
            <div>
              <h1 className="text-2xl sm:text-3xl font-bold">Smart Friend Core Brain</h1>
              <p className="text-sm text-muted-foreground flex items-center gap-2 mt-1">
                <Lock className="w-3 h-3" />
                আপনার সিস্টেম এখন রিয়েলটাইম মনিটরিং এর অধীনে
              </p>
            </div>
          </div>
          
          <div className="w-full sm:w-auto">
            <OnlineUsers />
          </div>
        </div>

        {/* Critical Alerts */}
        <CriticalNotifications />

        {/* AI Test Generator */}
        <GeneratedTests />

        {/* Pending Patches - Preview & Apply */}
        <PendingPatches />

        {/* Main Dashboard */}
        <Dashboard />

        {/* Patch History & Rollback */}
        <PatchHistory />

        {/* System Status */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Activity className="w-5 h-5" />
              System Status
            </CardTitle>
            <CardDescription>Real-time system health monitoring</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
              <div className="space-y-1">
                <p className="text-sm text-muted-foreground">API Status</p>
                <div className="flex items-center gap-2">
                  <div className={`w-2 h-2 rounded-full ${systemStatus.apiOnline ? 'bg-green-500' : 'bg-red-500'} animate-pulse`} />
                  <p className="font-semibold">{systemStatus.apiOnline ? 'Online' : 'Offline'}</p>
                </div>
              </div>
              <div className="space-y-1">
                <p className="text-sm text-muted-foreground flex items-center gap-1">
                  <Database className="w-3 h-3" />
                  Database
                </p>
                <p className="font-semibold capitalize">{systemStatus.databaseStatus}</p>
              </div>
              <div className="space-y-1">
                <p className="text-sm text-muted-foreground flex items-center gap-1">
                  <Clock className="w-3 h-3" />
                  Last Deploy
                </p>
                <p className="font-semibold text-sm">{systemStatus.lastDeploy}</p>
              </div>
              <div className="space-y-1">
                <p className="text-sm text-muted-foreground">Uptime</p>
                <p className="font-semibold">{systemStatus.uptime}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Project Scanner Card */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Scan className="w-5 h-5" />
              Project Scanner
            </CardTitle>
            <CardDescription>Scan codebase for errors, warnings, and optimization opportunities</CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex flex-col sm:flex-row gap-3">
              <Button 
                onClick={handleRunProjectScan}
                disabled={isScanning}
                className="flex-1"
              >
                {isScanning ? "Scanning..." : "Run Full Project Scan"}
              </Button>
              <Button 
                onClick={handleAnalyzeErrors}
                disabled={isAnalyzing}
                variant="outline"
                className="flex-1"
              >
                <Bug className="w-4 h-4 mr-2" />
                {isAnalyzing ? "Analyzing..." : "Analyze Error Logs"}
              </Button>
            </div>
            
            <div className="grid grid-cols-2 sm:grid-cols-4 gap-3 pt-2">
              <div className="p-3 rounded-lg bg-muted/30 text-center">
                <p className="text-2xl font-bold text-destructive">{scanResults.filter(r => r.severity === 'critical').length}</p>
                <p className="text-xs text-muted-foreground">Critical</p>
              </div>
              <div className="p-3 rounded-lg bg-muted/30 text-center">
                <p className="text-2xl font-bold text-orange-500">{scanResults.filter(r => r.severity === 'high').length}</p>
                <p className="text-xs text-muted-foreground">High</p>
              </div>
              <div className="p-3 rounded-lg bg-muted/30 text-center">
                <p className="text-2xl font-bold text-yellow-500">{scanResults.filter(r => r.severity === 'medium').length}</p>
                <p className="text-xs text-muted-foreground">Medium</p>
              </div>
              <div className="p-3 rounded-lg bg-muted/30 text-center">
                <p className="text-2xl font-bold text-blue-500">{scanResults.filter(r => r.severity === 'low').length}</p>
                <p className="text-xs text-muted-foreground">Low</p>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Core Brain Dashboard */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Sparkles className="w-5 h-5" />
              AI-Powered Analysis Dashboard
            </CardTitle>
            <CardDescription>Scan results, fix plans, and code patches</CardDescription>
          </CardHeader>
          <CardContent>
            <Tabs defaultValue="analytics" className="w-full">
              <TabsList className="grid w-full grid-cols-4">
                <TabsTrigger value="analytics">Analytics</TabsTrigger>
                <TabsTrigger value="scan">Scan Results</TabsTrigger>
                <TabsTrigger value="fixes">Fix Plans</TabsTrigger>
                <TabsTrigger value="patches">Code Patches</TabsTrigger>
              </TabsList>

              <TabsContent value="analytics" className="mt-4">
                <CoreBrainAnalytics />
              </TabsContent>

              <TabsContent value="scan" className="space-y-4 mt-4">
                <div className="flex justify-between items-center">
                  <p className="text-sm text-muted-foreground">
                    {scanResults.filter(r => !r.resolved).length} active issues
                  </p>
                </div>

                {scanResults.length === 0 ? (
                  <div className="text-center py-8 text-muted-foreground">
                    No scan results yet. Click "Run Full Project Scan" to start.
                  </div>
                ) : (
                  <div className="space-y-2">
                    {scanResults.filter(r => !r.resolved).map((result) => (
                      <div key={result.id} className="p-3 rounded-lg bg-muted/50 space-y-2">
                        <div className="flex items-center justify-between gap-2 flex-wrap">
                          <div className="flex items-center gap-2 flex-1">
                            <Badge variant={result.severity === 'critical' ? 'destructive' : 'default'}>
                              {result.severity}
                            </Badge>
                            <Badge variant="outline">{result.issue_type}</Badge>
                            <Badge variant="secondary">{result.scan_type}</Badge>
                          </div>
                          <div className="flex gap-2">
                            <Button 
                              size="sm" 
                              variant="outline"
                              onClick={() => handleGenerateFixPlan(result)}
                            >
                              Generate Fix
                            </Button>
                            <Button 
                              size="sm" 
                              variant="ghost"
                              onClick={() => handleMarkResolved(result.id)}
                            >
                              <CheckCircle2 className="w-4 h-4" />
                            </Button>
                          </div>
                        </div>
                        <p className="text-sm font-medium">{result.message}</p>
                        {result.file_path && (
                          <p className="text-xs text-muted-foreground font-mono">{result.file_path}</p>
                        )}
                      </div>
                    ))}
                  </div>
                )}
              </TabsContent>

              <TabsContent value="fixes" className="space-y-4 mt-4">
                <p className="text-sm text-muted-foreground">
                  {fixPlans.length} fix plans generated
                </p>

                {fixPlans.length === 0 ? (
                  <div className="text-center py-8 text-muted-foreground">
                    No fix plans yet. Generate fixes from scan results.
                  </div>
                ) : (
                  <div className="space-y-3">
                    {fixPlans.map((plan) => (
                      <div key={plan.id} className="p-4 rounded-lg bg-muted/50 space-y-3">
                        <div className="flex items-center justify-between gap-2 flex-wrap">
                          <h3 className="font-semibold">{plan.title}</h3>
                          <div className="flex items-center gap-2">
                            <Badge variant={plan.priority === 'critical' ? 'destructive' : 'default'}>
                              {plan.priority}
                            </Badge>
                            <Badge variant="outline">{plan.status}</Badge>
                          </div>
                        </div>
                        <p className="text-sm text-muted-foreground">{plan.description}</p>
                        
                        {plan.status === 'pending' && (
                          <div className="flex gap-2 flex-wrap">
                            <Button 
                              size="sm" 
                              onClick={() => handleApproveFixPlan(plan.id)}
                            >
                              Approve
                            </Button>
                            <Button 
                              size="sm" 
                              variant="outline"
                              onClick={() => handleRejectFixPlan(plan.id)}
                            >
                              Reject
                            </Button>
                            <Button 
                              size="sm" 
                              variant="secondary"
                              onClick={() => handleGenerateCodePatch(plan)}
                            >
                              <Code2 className="w-4 h-4 mr-2" />
                              Generate Patch
                            </Button>
                          </div>
                        )}
                      </div>
                    ))}
                  </div>
                )}
              </TabsContent>

              <TabsContent value="patches" className="space-y-4 mt-4">
                <p className="text-sm text-muted-foreground">
                  {codePatches.length} code patches generated
                </p>

                {codePatches.length === 0 ? (
                  <div className="text-center py-8 text-muted-foreground">
                    No code patches yet. Generate patches from approved fix plans.
                  </div>
                ) : (
                  <div className="space-y-3">
                    {codePatches.map((patch) => (
                      <div key={patch.id} className="p-4 rounded-lg bg-muted/50 space-y-3">
                        <div className="flex items-center justify-between gap-2">
                          <p className="font-mono text-sm">{patch.file_path}</p>
                          <Badge variant={patch.status === 'applied' ? 'secondary' : 'outline'}>
                            {patch.status}
                          </Badge>
                        </div>
                        <pre className="text-xs bg-background p-3 rounded overflow-x-auto max-h-64 overflow-y-auto">
                          <code>{patch.patched_code}</code>
                        </pre>
                        
                        {patch.status === 'pending' && (
                          <div className="flex gap-2">
                            <Button 
                              size="sm" 
                              variant="outline"
                              onClick={() => {
                                navigator.clipboard.writeText(patch.patched_code);
                                toast.success("Code copied to clipboard");
                              }}
                            >
                              Copy Code
                            </Button>
                          </div>
                        )}
                      </div>
                    ))}
                  </div>
                )}
              </TabsContent>
            </Tabs>
          </CardContent>
        </Card>

        {/* Error Logs */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertCircle className="w-5 h-5" />
              Error Logs
            </CardTitle>
            <CardDescription>Latest 20 system errors and warnings</CardDescription>
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <div className="text-center py-8 text-muted-foreground">
                Loading logs...
              </div>
            ) : logs.length === 0 ? (
              <div className="text-center py-8 text-muted-foreground">
                No logs found
              </div>
            ) : (
              <div className="space-y-3">
                {logs.map((log) => (
                  <div key={log.id} className="flex flex-col sm:flex-row sm:items-center justify-between gap-2 p-3 rounded-lg bg-muted/50 hover:bg-muted transition-colors">
                    <div className="flex-1 space-y-1">
                      <div className="flex items-center gap-2 flex-wrap">
                        <Badge variant={getSeverityColor(log.type)} className="text-xs">
                          {log.type}
                        </Badge>
                        <span className="text-xs text-muted-foreground">{formatDate(log.created_at)}</span>
                      </div>
                      <p className="text-sm">{log.message}</p>
                      {log.details && (
                        <p className="text-xs text-muted-foreground font-mono">
                          {JSON.stringify(log.details)}
                        </p>
                      )}
                    </div>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>

        {/* Module Generator */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Wand2 className="w-5 h-5" />
              AI Module Generator
            </CardTitle>
            <CardDescription>Generate complete modules with routes, components, and database schema</CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="space-y-2">
              <label className="text-sm font-medium">Module Name</label>
              <input
                type="text"
                placeholder="e.g., Smart Pay, Smart Doctor, Smart Lawyer"
                value={moduleName}
                onChange={(e) => setModuleName(e.target.value)}
                className="w-full px-3 py-2 rounded-md border border-input bg-background"
              />
            </div>

            <div className="space-y-2">
              <label className="text-sm font-medium">Description</label>
              <Textarea
                placeholder="Describe what this module should do..."
                value={moduleDescription}
                onChange={(e) => setModuleDescription(e.target.value)}
                className="min-h-24 resize-none"
              />
            </div>

            <div className="space-y-2">
              <label className="text-sm font-medium">Features (one per field)</label>
              {moduleFeatures.map((feature, index) => (
                <div key={index} className="flex gap-2">
                  <input
                    type="text"
                    placeholder={`Feature ${index + 1}: e.g., Send money, View transactions`}
                    value={feature}
                    onChange={(e) => updateFeature(index, e.target.value)}
                    className="flex-1 px-3 py-2 rounded-md border border-input bg-background"
                  />
                  {moduleFeatures.length > 1 && (
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => removeFeature(index)}
                    >
                      Remove
                    </Button>
                  )}
                </div>
              ))}
              <Button
                size="sm"
                variant="outline"
                onClick={addFeatureField}
                className="w-full"
              >
                Add Feature
              </Button>
            </div>

            <Button 
              onClick={handleGenerateModule} 
              disabled={isGeneratingModule}
              className="w-full"
            >
              <Package className="w-4 h-4 mr-2" />
              {isGeneratingModule ? "Generating..." : "Generate Module Scaffold"}
            </Button>

            {moduleScaffold && (
              <div className="mt-6 space-y-4">
                <div className="flex items-center justify-between">
                  <h3 className="text-lg font-semibold">Generated Scaffold</h3>
                  <Badge variant="secondary">Ready to Review</Badge>
                </div>

                <Tabs defaultValue="routes" className="w-full">
                  <TabsList className="grid w-full grid-cols-4">
                    <TabsTrigger value="routes">Routes</TabsTrigger>
                    <TabsTrigger value="components">Components</TabsTrigger>
                    <TabsTrigger value="database">Database</TabsTrigger>
                    <TabsTrigger value="functions">Functions</TabsTrigger>
                  </TabsList>

                  <TabsContent value="routes" className="space-y-3 mt-4">
                    {moduleScaffold.routes?.map((route, idx) => (
                      <div key={idx} className="p-3 rounded-lg bg-muted/50 space-y-2">
                        <div className="flex items-center justify-between">
                          <p className="font-mono text-sm font-medium">{route.path}</p>
                          <Badge variant="outline">{route.component}</Badge>
                        </div>
                        <pre className="text-xs bg-background p-2 rounded overflow-x-auto max-h-40 overflow-y-auto">
                          <code>{route.code}</code>
                        </pre>
                      </div>
                    ))}
                  </TabsContent>

                  <TabsContent value="components" className="space-y-3 mt-4">
                    {moduleScaffold.components?.map((comp, idx) => (
                      <div key={idx} className="p-3 rounded-lg bg-muted/50 space-y-2">
                        <div className="flex items-center justify-between">
                          <p className="font-medium text-sm">{comp.name}</p>
                          <p className="font-mono text-xs text-muted-foreground">{comp.path}</p>
                        </div>
                        <pre className="text-xs bg-background p-2 rounded overflow-x-auto max-h-40 overflow-y-auto">
                          <code>{comp.code}</code>
                        </pre>
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => {
                            navigator.clipboard.writeText(comp.code);
                            toast.success("Component code copied");
                          }}
                        >
                          Copy Code
                        </Button>
                      </div>
                    ))}
                  </TabsContent>

                  <TabsContent value="database" className="space-y-3 mt-4">
                    {moduleScaffold.database?.map((db, idx) => (
                      <div key={idx} className="p-3 rounded-lg bg-muted/50 space-y-2">
                        <p className="font-medium text-sm">Migration {idx + 1}</p>
                        <pre className="text-xs bg-background p-2 rounded overflow-x-auto max-h-40 overflow-y-auto">
                          <code>{db.migrations}</code>
                        </pre>
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => {
                            navigator.clipboard.writeText(db.migrations);
                            toast.success("Migration SQL copied");
                          }}
                        >
                          Copy SQL
                        </Button>
                      </div>
                    ))}
                  </TabsContent>

                  <TabsContent value="functions" className="space-y-3 mt-4">
                    {moduleScaffold.edgeFunctions?.map((func, idx) => (
                      <div key={idx} className="p-3 rounded-lg bg-muted/50 space-y-2">
                        <p className="font-mono text-sm font-medium">{func.name}</p>
                        <pre className="text-xs bg-background p-2 rounded overflow-x-auto max-h-40 overflow-y-auto">
                          <code>{func.code}</code>
                        </pre>
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => {
                            navigator.clipboard.writeText(func.code);
                            toast.success("Function code copied");
                          }}
                        >
                          Copy Code
                        </Button>
                      </div>
                    ))}
                  </TabsContent>
                </Tabs>

                <div className="flex gap-2 pt-2">
                  <Button
                    variant="outline"
                    onClick={() => {
                      setModuleScaffold(null);
                      setModuleName("");
                      setModuleDescription("");
                      setModuleFeatures([""]);
                      toast.success("Generator reset");
                    }}
                  >
                    Generate Another
                  </Button>
                </div>
              </div>
            )}
          </CardContent>
        </Card>

        {/* Upgrade Planner */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FileText className="w-5 h-5" />
              Upgrade Planner
            </CardTitle>
            <CardDescription>Write and save upgrade ideas for Smart Friend</CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="space-y-2">
              <label className="text-sm font-medium">Plan Title</label>
              <input
                type="text"
                placeholder="e.g., Implement Bengali voice recognition"
                value={planTitle}
                onChange={(e) => setPlanTitle(e.target.value)}
                className="w-full px-3 py-2 rounded-md border border-input bg-background"
              />
            </div>
            <div className="space-y-2">
              <label className="text-sm font-medium">Description</label>
              <Textarea
                placeholder="Write your upgrade idea details here... (goals, requirements, expected outcome...)"
                value={upgradeIdea}
                onChange={(e) => setUpgradeIdea(e.target.value)}
                className="min-h-32 resize-none"
              />
            </div>
            <Button onClick={handleSaveUpgrade} className="w-full sm:w-auto">
              Save Upgrade Plan
            </Button>

            {plans.length > 0 && (
              <div className="mt-6 space-y-2">
                <h3 className="text-sm font-medium">Recent Plans</h3>
                <div className="space-y-2">
                  {plans.slice(0, 5).map((plan) => (
                    <div key={plan.id} className="p-3 rounded-lg bg-muted/50 space-y-1">
                      <div className="flex items-center justify-between gap-2">
                        <p className="font-medium text-sm">{plan.title}</p>
                        <Badge variant="secondary" className="text-xs">{plan.status}</Badge>
                      </div>
                      <p className="text-xs text-muted-foreground line-clamp-2">{plan.description}</p>
                      <p className="text-xs text-muted-foreground">{formatDate(plan.created_at)}</p>
                    </div>
                  ))}
                </div>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </div>
  );
};

export default CoreBrain;
