import { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useLanguage } from "@/hooks/useLanguage";
import { useToast } from "@/hooks/use-toast";
import { Globe, DollarSign, MapPin } from "lucide-react";

const CountrySettings = () => {
  const { t, language } = useLanguage();
  const { toast } = useToast();
  const [country, setCountry] = useState("BD");
  const [currency, setCurrency] = useState("BDT");

  const countries = [
    { code: "BD", name: "Bangladesh", nameBn: "বাংলাদেশ", currency: "BDT", flag: "🇧🇩" },
    { code: "IN", name: "India", nameBn: "ভারত", currency: "INR", flag: "🇮🇳" },
    { code: "PK", name: "Pakistan", nameBn: "পাকিস্তান", currency: "PKR", flag: "🇵🇰" },
    { code: "AE", name: "UAE", nameBn: "সংযুক্ত আরব আমিরাত", currency: "AED", flag: "🇦🇪" },
    { code: "SA", name: "Saudi Arabia", nameBn: "সৌদি আরব", currency: "SAR", flag: "🇸🇦" },
    { code: "MY", name: "Malaysia", nameBn: "মালয়েশিয়া", currency: "MYR", flag: "🇲🇾" },
    { code: "ID", name: "Indonesia", nameBn: "ইন্দোনেশিয়া", currency: "IDR", flag: "🇮🇩" },
    { code: "US", name: "United States", nameBn: "যুক্তরাষ্ট্র", currency: "USD", flag: "🇺🇸" },
    { code: "GB", name: "United Kingdom", nameBn: "যুক্তরাজ্য", currency: "GBP", flag: "🇬🇧" },
    { code: "CA", name: "Canada", nameBn: "কানাডা", currency: "CAD", flag: "🇨🇦" },
  ];

  const handleSave = () => {
    localStorage.setItem("user_country", country);
    localStorage.setItem("user_currency", currency);
    toast({
      title: language === "bn" ? "সংরক্ষিত" : "Saved",
      description: language === "bn" 
        ? "দেশ এবং মুদ্রা সেটিংস আপডেট হয়েছে" 
        : "Country and currency settings updated"
    });
  };

  return (
    <div className="min-h-screen p-6 max-w-2xl mx-auto pb-24">
      <div className="space-y-6">
        <div>
          <h1 className="text-3xl font-bold">
            {language === "bn" ? "দেশ ও অঞ্চল" : "Country & Region"}
          </h1>
          <p className="text-muted-foreground mt-2">
            {language === "bn" 
              ? "আপনার দেশ এবং মুদ্রা সেটিংস কনফিগার করুন" 
              : "Configure your country and currency settings"}
          </p>
        </div>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <MapPin className="h-5 w-5" />
              {language === "bn" ? "দেশ নির্বাচন" : "Select Country"}
            </CardTitle>
            <CardDescription>
              {language === "bn"
                ? "আপনার দেশ অনুযায়ী কন্টেন্ট এবং সেবা কাস্টমাইজ হবে"
                : "Content and services will be customized for your country"}
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <Select value={country} onValueChange={(val) => {
              setCountry(val);
              const selectedCountry = countries.find(c => c.code === val);
              if (selectedCountry) setCurrency(selectedCountry.currency);
            }}>
              <SelectTrigger>
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                {countries.map((c) => (
                  <SelectItem key={c.code} value={c.code}>
                    <span className="flex items-center gap-2">
                      <span>{c.flag}</span>
                      <span>{language === "bn" ? c.nameBn : c.name}</span>
                    </span>
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <DollarSign className="h-5 w-5" />
              {language === "bn" ? "মুদ্রা" : "Currency"}
            </CardTitle>
            <CardDescription>
              {language === "bn"
                ? "মূল্য এই মুদ্রায় দেখানো হবে"
                : "Prices will be displayed in this currency"}
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="p-4 rounded-lg bg-muted/50">
              <div className="flex items-center justify-between">
                <span className="font-medium">{language === "bn" ? "নির্বাচিত মুদ্রা" : "Selected Currency"}</span>
                <span className="text-2xl font-bold text-primary">{currency}</span>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Globe className="h-5 w-5" />
              {language === "bn" ? "এলাকা-ভিত্তিক বৈশিষ্ট্য" : "Region-based Features"}
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-3">
            <div className="flex items-center justify-between p-3 rounded-lg bg-muted/50">
              <span className="text-sm">{language === "bn" ? "স্থানীয় সংবাদ" : "Local News"}</span>
              <span className="text-xs text-primary">✓ {language === "bn" ? "সক্রিয়" : "Active"}</span>
            </div>
            <div className="flex items-center justify-between p-3 rounded-lg bg-muted/50">
              <span className="text-sm">{language === "bn" ? "স্থানীয় চাকরি" : "Local Jobs"}</span>
              <span className="text-xs text-primary">✓ {language === "bn" ? "সক্রিয়" : "Active"}</span>
            </div>
            <div className="flex items-center justify-between p-3 rounded-lg bg-muted/50">
              <span className="text-sm">{language === "bn" ? "পর্যটন প্যাকেজ" : "Tourism Packages"}</span>
              <span className="text-xs text-primary">✓ {language === "bn" ? "সক্রিয়" : "Active"}</span>
            </div>
            <div className="flex items-center justify-between p-3 rounded-lg bg-muted/50">
              <span className="text-sm">{language === "bn" ? "স্থানীয় পেমেন্ট" : "Local Payments"}</span>
              <span className="text-xs text-primary">✓ {language === "bn" ? "সক্রিয়" : "Active"}</span>
            </div>
          </CardContent>
        </Card>

        <Button onClick={handleSave} className="w-full" size="lg">
          {language === "bn" ? "সংরক্ষণ করুন" : "Save Settings"}
        </Button>
      </div>
    </div>
  );
};

export default CountrySettings;
