import { useState, useEffect } from "react";
import { OwnerGuard } from "@/components/OwnerGuard";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { useProductionExport } from "@/hooks/useProductionExport";
import { Download, Rocket, Package, Server, FileText, Zap, AlertCircle, CheckCircle2, XCircle } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";

const DeploymentCenter = () => {
  const { toast } = useToast();
  const { exportProduction, exporting } = useProductionExport("smart-friend-project");
  const [healthStatus, setHealthStatus] = useState({
    supabaseUrl: false,
    supabaseKey: false,
    readyToDeploy: false,
    warnings: [] as string[]
  });

  useEffect(() => {
    const supabaseUrl = import.meta.env.VITE_SUPABASE_URL;
    const supabaseKey = import.meta.env.VITE_SUPABASE_PUBLISHABLE_KEY;
    
    const warnings: string[] = [];
    
    if (!supabaseUrl || supabaseUrl.includes('your-project')) {
      warnings.push("Supabase URL not configured");
    }
    
    if (!supabaseKey || supabaseKey.includes('your-key')) {
      warnings.push("Supabase key not configured");
    }
    
    setHealthStatus({
      supabaseUrl: !!supabaseUrl && !supabaseUrl.includes('your-project'),
      supabaseKey: !!supabaseKey && !supabaseKey.includes('your-key'),
      readyToDeploy: warnings.length === 0,
      warnings
    });
  }, []);

  const handleExportProduction = async () => {
    const result = await exportProduction("smart-friend");
    
    if (result) {
      toast({
        title: "All tasks completed. Production ZIP is ready.",
        description: `Package exported successfully (${result.size}, ${result.files} files)`,
      });
    }
  };

  return (
    <OwnerGuard>
      <div className="min-h-screen bg-background p-6">
        <div className="max-w-4xl mx-auto space-y-6">
          <div className="flex items-center gap-3">
            <Rocket className="h-8 w-8 text-primary" />
            <div>
              <h1 className="text-3xl font-bold text-foreground">Deployment Center</h1>
              <p className="text-muted-foreground">Export and deploy your application</p>
            </div>
          </div>

          {/* Health Status */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                {healthStatus.readyToDeploy ? (
                  <CheckCircle2 className="h-5 w-5 text-green-500" />
                ) : (
                  <AlertCircle className="h-5 w-5 text-yellow-500" />
                )}
                System Health Check
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <div className="flex items-center justify-between">
                  <span className="text-sm">Supabase Backend</span>
                  {healthStatus.supabaseUrl && healthStatus.supabaseKey ? (
                    <Badge variant="default" className="bg-green-500">
                      <CheckCircle2 className="h-3 w-3 mr-1" />
                      Configured
                    </Badge>
                  ) : (
                    <Badge variant="destructive">
                      <XCircle className="h-3 w-3 mr-1" />
                      Not Configured
                    </Badge>
                  )}
                </div>
                
                <div className="flex items-center justify-between">
                  <span className="text-sm">Passwordless Auth</span>
                  <Badge variant="default" className="bg-green-500">
                    <CheckCircle2 className="h-3 w-3 mr-1" />
                    Enabled (OTP)
                  </Badge>
                </div>
                
                <div className="flex items-center justify-between">
                  <span className="text-sm">Build System</span>
                  <Badge variant="default" className="bg-green-500">
                    <CheckCircle2 className="h-3 w-3 mr-1" />
                    Ready
                  </Badge>
                </div>
              </div>

              {healthStatus.warnings.length > 0 && (
                <Alert variant="destructive">
                  <AlertCircle className="h-4 w-4" />
                  <AlertTitle>Configuration Warnings</AlertTitle>
                  <AlertDescription>
                    <ul className="list-disc list-inside space-y-1 mt-2">
                      {healthStatus.warnings.map((warning, i) => (
                        <li key={i} className="text-sm">{warning}</li>
                      ))}
                    </ul>
                    <p className="text-sm mt-2">
                      Note: You can still export, but backend features won't work without Supabase configuration.
                    </p>
                  </AlertDescription>
                </Alert>
              )}
            </CardContent>
          </Card>

          <Alert>
            <Package className="h-4 w-4" />
            <AlertDescription>
              Zero-config production package with complete deployment guides for cPanel, Netlify, and Vercel. Just upload and go live! 🚀
            </AlertDescription>
          </Alert>

          {/* Export Production Package */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Download className="h-5 w-5" />
                Export Zero-Config Production Package
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-3">
                <div className="flex items-start gap-3 p-3 bg-primary/5 rounded-lg">
                  <Zap className="h-5 w-5 text-primary mt-0.5" />
                  <div>
                    <h4 className="font-semibold text-sm">What's Included:</h4>
                    <ul className="text-sm text-muted-foreground list-disc list-inside space-y-1 mt-1">
                      <li>Complete deployment guides (cPanel, Netlify, Vercel)</li>
                      <li>SPA routing configuration (.htaccess, redirects)</li>
                      <li>Environment templates (pre-filled with your config)</li>
                      <li>Security headers & caching rules</li>
                      <li>Build instructions & health check guide</li>
                      <li>Quick start instructions</li>
                    </ul>
                  </div>
                </div>

                <div className="flex items-start gap-3 p-3 bg-secondary/50 rounded-lg">
                  <FileText className="h-5 w-5 text-secondary-foreground mt-0.5" />
                  <div>
                    <h4 className="font-semibold text-sm">Deployment Guides:</h4>
                    <ul className="text-sm text-muted-foreground space-y-1 mt-1">
                      <li>📄 README-DEPLOY.md - Main deployment guide</li>
                      <li>📄 BUILD-INSTRUCTIONS.md - How to build</li>
                      <li>📄 HEALTH-CHECK.md - Deployment checklist</li>
                      <li>📄 DEPLOY-CPANEL.md - cPanel instructions</li>
                      <li>📄 DEPLOY-NETLIFY.md - Netlify guide</li>
                      <li>📄 DEPLOY-VERCEL.md - Vercel guide</li>
                      <li>📄 QUICKSTART.md - Quick reference</li>
                    </ul>
                  </div>
                </div>

                <div className="bg-yellow-500/10 border border-yellow-500/20 rounded-lg p-3">
                  <h4 className="font-semibold text-sm flex items-center gap-2 text-yellow-700 dark:text-yellow-400">
                    <AlertCircle className="h-4 w-4" />
                    Build Required
                  </h4>
                  <p className="text-sm text-muted-foreground mt-2">
                    After downloading, you must build locally:
                  </p>
                  <ol className="text-sm text-muted-foreground list-decimal list-inside space-y-1 mt-2">
                    <li>Run <code className="bg-muted px-1 py-0.5 rounded">npm install</code></li>
                    <li>Run <code className="bg-muted px-1 py-0.5 rounded">npm run build</code></li>
                    <li>Upload <code className="bg-muted px-1 py-0.5 rounded">dist</code> folder contents to hosting</li>
                    <li>Add config files from this ZIP</li>
                  </ol>
                </div>
              </div>

              <Button
                onClick={handleExportProduction}
                disabled={exporting}
                size="lg"
                className="w-full"
              >
                {exporting ? (
                  <>
                    <Zap className="h-5 w-5 mr-2 animate-spin" />
                    Building Package...
                  </>
                ) : (
                  <>
                    <Download className="h-5 w-5 mr-2" />
                    Download Production Configuration Package
                  </>
                )}
              </Button>

              <p className="text-xs text-muted-foreground text-center">
                Includes passwordless auth (OTP), deployment configs, and complete documentation
              </p>
            </CardContent>
          </Card>

          {/* Deployment Options */}
          <div className="grid gap-4 md:grid-cols-3">
            <Card>
              <CardHeader>
                <CardTitle className="text-lg flex items-center gap-2">
                  <Server className="h-5 w-5" />
                  cPanel
                </CardTitle>
              </CardHeader>
              <CardContent>
                <Badge variant="secondary">Shared Hosting</Badge>
                <p className="text-xs text-muted-foreground mt-2">
                  Upload dist folder to public_html
                </p>
                <p className="text-xs text-green-600 mt-1 font-semibold">
                  ✅ Easiest for beginners
                </p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle className="text-lg flex items-center gap-2">
                  <Server className="h-5 w-5" />
                  Netlify
                </CardTitle>
              </CardHeader>
              <CardContent>
                <Badge>Auto-Deploy</Badge>
                <p className="text-xs text-muted-foreground mt-2">
                  Deploy via Git or drag & drop
                </p>
                <p className="text-xs text-green-600 mt-1 font-semibold">
                  ✅ Best for continuous deployment
                </p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle className="text-lg flex items-center gap-2">
                  <Server className="h-5 w-5" />
                  Vercel
                </CardTitle>
              </CardHeader>
              <CardContent>
                <Badge>Edge Network</Badge>
                <p className="text-xs text-muted-foreground mt-2">
                  Import from GitHub repository
                </p>
                <p className="text-xs text-green-600 mt-1 font-semibold">
                  ✅ Best performance globally
                </p>
              </CardContent>
            </Card>
          </div>

          {/* Final Instructions */}
          <Alert>
            <CheckCircle2 className="h-4 w-4" />
            <AlertTitle>What's Next?</AlertTitle>
            <AlertDescription>
              <ol className="list-decimal list-inside space-y-1 text-sm mt-2">
                <li>Download the configuration package above</li>
                <li>Extract the ZIP file</li>
                <li>Follow BUILD-INSTRUCTIONS.md to create production build</li>
                <li>Follow platform-specific guide (DEPLOY-CPANEL.md, etc.)</li>
                <li>Upload your built files + configs to hosting</li>
                <li>Use HEALTH-CHECK.md to verify deployment</li>
              </ol>
            </AlertDescription>
          </Alert>
        </div>
      </div>
    </OwnerGuard>
  );
};

export default DeploymentCenter;
