import { useNavigate } from "react-router-dom";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ArrowLeft, Rocket, Code, Cloud, Database, Zap, GitBranch } from "lucide-react";
import { useLanguage } from "@/hooks/useLanguage";

const FutureFeatures = () => {
  const navigate = useNavigate();
  const { language } = useLanguage();

  const features = [
    {
      title: language === 'bn' ? 'স্বয়ংক্রিয় DevOps' : 'Auto DevOps',
      titleBn: 'স্বয়ংক্রিয় DevOps',
      description: language === 'bn' 
        ? 'CI/CD পাইপলাইন, স্বয়ংক্রিয় ডিপ্লয়মেন্ট এবং মনিটরিং'
        : 'CI/CD pipelines, automated deployments, and monitoring',
      icon: GitBranch,
      status: 'roadmap',
      gradient: 'from-blue-500 to-cyan-500'
    },
    {
      title: language === 'bn' ? 'ক্লাউড স্কেলিং' : 'Cloud Scaling',
      titleBn: 'ক্লাউড স্কেলিং',
      description: language === 'bn'
        ? 'স্বয়ংক্রিয় স্কেলিং এবং লোড ব্যালান্সিং'
        : 'Automatic scaling and load balancing',
      icon: Cloud,
      status: 'planned',
      gradient: 'from-purple-500 to-pink-500'
    },
    {
      title: language === 'bn' ? 'ডাটাবেস অপ্টিমাইজেশন' : 'Database Optimization',
      titleBn: 'ডাটাবেস অপ্টিমাইজেশন',
      description: language === 'bn'
        ? 'স্বয়ংক্রিয় ইন্ডেক্সিং এবং কোয়েরি অপ্টিমাইজেশন'
        : 'Automatic indexing and query optimization',
      icon: Database,
      status: 'planned',
      gradient: 'from-green-500 to-emerald-500'
    },
    {
      title: language === 'bn' ? 'পারফরম্যান্স মনিটরিং' : 'Performance Monitoring',
      titleBn: 'পারফরম্যান্স মনিটরিং',
      description: language === 'bn'
        ? 'রিয়েল-টাইম পারফরম্যান্স ট্র্যাকিং এবং অ্যালার্ট'
        : 'Real-time performance tracking and alerts',
      icon: Zap,
      status: 'roadmap',
      gradient: 'from-orange-500 to-red-500'
    },
    {
      title: language === 'bn' ? 'Smart School API' : 'Smart School API',
      titleBn: 'Smart School API',
      description: language === 'bn'
        ? 'Laravel ব্যাকএন্ড ইন্টিগ্রেশন'
        : 'Laravel backend integration',
      icon: Code,
      status: 'in_progress',
      gradient: 'from-indigo-500 to-purple-500'
    },
    {
      title: language === 'bn' ? 'Smart Pay সংযোগ' : 'Smart Pay Integration',
      titleBn: 'Smart Pay সংযোগ',
      description: language === 'bn'
        ? 'পেমেন্ট গেটওয়ে এবং ওয়ালেট ম্যানেজমেন্ট'
        : 'Payment gateway and wallet management',
      icon: Rocket,
      status: 'in_progress',
      gradient: 'from-pink-500 to-rose-500'
    },
  ];

  const getStatusBadge = (status: string) => {
    switch (status) {
      case 'in_progress':
        return <Badge variant="default">{language === 'bn' ? 'চলমান' : 'In Progress'}</Badge>;
      case 'planned':
        return <Badge variant="secondary">{language === 'bn' ? 'পরিকল্পিত' : 'Planned'}</Badge>;
      case 'roadmap':
        return <Badge variant="outline">{language === 'bn' ? 'রোডম্যাপে' : 'Roadmap'}</Badge>;
      default:
        return null;
    }
  };

  return (
    <div className="h-full overflow-auto pb-20">
      <div className="p-6 max-w-4xl mx-auto space-y-6">
        {/* Header */}
        <div className="flex items-center gap-3">
          <Button variant="ghost" size="icon" onClick={() => navigate(-1)}>
            <ArrowLeft className="h-5 w-5" />
          </Button>
          <div>
            <h1 className="text-2xl font-bold">
              {language === 'bn' ? 'আসছে শীঘ্রই' : 'Coming Soon'}
            </h1>
            <p className="text-sm text-muted-foreground">
              {language === 'bn' 
                ? 'ভবিষ্যত বৈশিষ্ট্য এবং পরিকল্পনা'
                : 'Future features and roadmap'}
            </p>
          </div>
        </div>

        {/* Features Grid */}
        <div className="grid md:grid-cols-2 gap-4">
          {features.map((feature, index) => {
            const Icon = feature.icon;
            return (
              <Card key={index} className="hover:shadow-lg transition-shadow">
                <CardHeader>
                  <div className="flex items-start justify-between">
                    <div className={`p-3 rounded-lg bg-gradient-to-br ${feature.gradient}`}>
                      <Icon className="h-6 w-6 text-white" />
                    </div>
                    {getStatusBadge(feature.status)}
                  </div>
                  <CardTitle className="mt-4">{feature.title}</CardTitle>
                  <CardDescription>{feature.description}</CardDescription>
                </CardHeader>
                <CardContent>
                  <Button variant="outline" size="sm" disabled className="w-full">
                    {language === 'bn' ? 'শীঘ্রই আসছে' : 'Coming Soon'}
                  </Button>
                </CardContent>
              </Card>
            );
          })}
        </div>

        {/* Info Card */}
        <Card className="bg-gradient-to-br from-primary/5 to-secondary/5 border-primary/10">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Rocket className="h-5 w-5 text-primary" />
              {language === 'bn' ? 'উন্নয়ন চলছে' : 'Under Development'}
            </CardTitle>
            <CardDescription>
              {language === 'bn'
                ? 'এই বৈশিষ্ট্যগুলি সক্রিয়ভাবে উন্নত করা হচ্ছে। আপডেটের জন্য সংযুক্ত থাকুন!'
                : 'These features are being actively developed. Stay tuned for updates!'}
            </CardDescription>
          </CardHeader>
        </Card>
      </div>
    </div>
  );
};

export default FutureFeatures;
