import { useQuery } from "@tanstack/react-query";
import { supabase } from "@/integrations/supabase/client";
import { Link } from "react-router-dom";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Progress } from "@/components/ui/progress";
import { ArrowLeft, Edit, Eye, EyeOff, Lock, Unlock, Plus } from "lucide-react";
import { useState } from "react";

export default function HQDashboard() {
  const [categoryFilter, setCategoryFilter] = useState<string>("all");
  const [statusFilter, setStatusFilter] = useState<string>("all");

  const { data: modules, isLoading } = useQuery({
    queryKey: ["hq-modules", categoryFilter, statusFilter],
    queryFn: async () => {
      let query = supabase
        .from("seba_modules")
        .select("*")
        .order("name");

      if (categoryFilter !== "all") {
        query = query.eq("category", categoryFilter);
      }
      if (statusFilter !== "all") {
        query = query.eq("status", statusFilter);
      }

      const { data, error } = await query;
      if (error) throw error;
      return data;
    },
  });

  const categories = Array.from(new Set(modules?.map(m => m.category) || []));

  const getStatusColor = (status: string) => {
    switch (status) {
      case "live": return "bg-green-500/10 text-green-500 border-green-500/20";
      case "in-progress": return "bg-blue-500/10 text-blue-500 border-blue-500/20";
      case "planned": return "bg-gray-500/10 text-gray-500 border-gray-500/20";
      default: return "bg-gray-500/10 text-gray-500 border-gray-500/20";
    }
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-background via-primary/5 to-secondary/5">
      <div className="container py-8">
        <div className="space-y-8">
          {/* Header */}
          <div className="flex items-center justify-between">
            <div>
              <div className="flex items-center gap-2 mb-2">
                <Button variant="ghost" size="sm" asChild>
                  <Link to="/">
                    <ArrowLeft className="mr-2 h-4 w-4" />
                    Public Site
                  </Link>
                </Button>
              </div>
              <h1 className="text-4xl font-bold">Smart Seba HQ</h1>
              <p className="text-muted-foreground mt-2">Module Management Dashboard</p>
            </div>
            <Button asChild>
              <Link to="/hq/modules/new">
                <Plus className="mr-2 h-4 w-4" />
                Add New Module
              </Link>
            </Button>
          </div>

          {/* Stats Cards */}
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <Card>
              <CardHeader className="pb-3">
                <CardDescription>Total Modules</CardDescription>
                <CardTitle className="text-3xl">{modules?.length || 0}</CardTitle>
              </CardHeader>
            </Card>
            <Card>
              <CardHeader className="pb-3">
                <CardDescription>Live</CardDescription>
                <CardTitle className="text-3xl text-green-500">
                  {modules?.filter(m => m.status === "live").length || 0}
                </CardTitle>
              </CardHeader>
            </Card>
            <Card>
              <CardHeader className="pb-3">
                <CardDescription>In Progress</CardDescription>
                <CardTitle className="text-3xl text-blue-500">
                  {modules?.filter(m => m.status === "in-progress").length || 0}
                </CardTitle>
              </CardHeader>
            </Card>
            <Card>
              <CardHeader className="pb-3">
                <CardDescription>Planned</CardDescription>
                <CardTitle className="text-3xl text-gray-500">
                  {modules?.filter(m => m.status === "planned").length || 0}
                </CardTitle>
              </CardHeader>
            </Card>
          </div>

          {/* Filters */}
          <Card>
            <CardHeader>
              <CardTitle>Filters</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="flex gap-4 flex-wrap">
                <div className="flex-1 min-w-[200px]">
                  <Select value={categoryFilter} onValueChange={setCategoryFilter}>
                    <SelectTrigger>
                      <SelectValue placeholder="Filter by category" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">All Categories</SelectItem>
                      {categories.map(cat => (
                        <SelectItem key={cat} value={cat}>{cat}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div className="flex-1 min-w-[200px]">
                  <Select value={statusFilter} onValueChange={setStatusFilter}>
                    <SelectTrigger>
                      <SelectValue placeholder="Filter by status" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">All Status</SelectItem>
                      <SelectItem value="live">Live</SelectItem>
                      <SelectItem value="in-progress">In Progress</SelectItem>
                      <SelectItem value="planned">Planned</SelectItem>
                      <SelectItem value="paused">Paused</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Modules Table */}
          <Card>
            <CardHeader>
              <CardTitle>All Modules</CardTitle>
              <CardDescription>
                Manage all Smart Seba modules from this dashboard
              </CardDescription>
            </CardHeader>
            <CardContent>
              {isLoading ? (
                <div className="space-y-4">
                  {[...Array(5)].map((_, i) => (
                    <div key={i} className="h-16 bg-muted animate-pulse rounded" />
                  ))}
                </div>
              ) : (
                <div className="overflow-x-auto">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead className="w-[50px]">Icon</TableHead>
                        <TableHead>Name</TableHead>
                        <TableHead>Category</TableHead>
                        <TableHead>Status</TableHead>
                        <TableHead>Progress</TableHead>
                        <TableHead className="text-center">Visibility</TableHead>
                        <TableHead className="text-center">Public</TableHead>
                        <TableHead className="text-right">Actions</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {modules?.map((module) => (
                        <TableRow key={module.id}>
                          <TableCell className="text-2xl">{module.icon}</TableCell>
                          <TableCell>
                            <div>
                              <div className="font-medium">{module.name}</div>
                              {module.name_bn && (
                                <div className="text-xs text-muted-foreground">{module.name_bn}</div>
                              )}
                            </div>
                          </TableCell>
                          <TableCell>
                            <Badge variant="secondary">{module.category}</Badge>
                          </TableCell>
                          <TableCell>
                            <Badge className={getStatusColor(module.status)} variant="outline">
                              {module.status}
                            </Badge>
                          </TableCell>
                          <TableCell>
                            <div className="flex items-center gap-2 min-w-[120px]">
                              <Progress value={module.progress_percent} className="h-2" />
                              <span className="text-xs text-muted-foreground whitespace-nowrap">
                                {module.progress_percent}%
                              </span>
                            </div>
                          </TableCell>
                          <TableCell className="text-center">
                            {module.is_visible_public ? (
                              <Eye className="h-4 w-4 text-green-500 mx-auto" />
                            ) : (
                              <EyeOff className="h-4 w-4 text-gray-500 mx-auto" />
                            )}
                          </TableCell>
                          <TableCell className="text-center">
                            {module.is_public_unlocked ? (
                              <Unlock className="h-4 w-4 text-green-500 mx-auto" />
                            ) : (
                              <Lock className="h-4 w-4 text-amber-500 mx-auto" />
                            )}
                          </TableCell>
                          <TableCell className="text-right">
                            <div className="flex gap-2 justify-end">
                              <Button variant="outline" size="sm" asChild>
                                <Link to={`/module/${module.slug}`}>View</Link>
                              </Button>
                              <Button size="sm" asChild>
                                <Link to={`/hq/modules/${module.id}`}>
                                  <Edit className="h-3 w-3" />
                                </Link>
                              </Button>
                            </div>
                          </TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
