import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { useParams, useNavigate, Link } from "react-router-dom";
import { supabase } from "@/integrations/supabase/client";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import { Slider } from "@/components/ui/slider";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Separator } from "@/components/ui/separator";
import { ArrowLeft, Save, Trash2 } from "lucide-react";
import { toast } from "sonner";
import { useState, useEffect } from "react";

export default function HQModuleEdit() {
  const { id } = useParams<{ id: string }>();
  const navigate = useNavigate();
  const queryClient = useQueryClient();
  const isNew = id === "new";

  const [formData, setFormData] = useState({
    slug: "",
    name: "",
    name_bn: "",
    short_tagline: "",
    short_tagline_bn: "",
    category: "",
    icon: "📦",
    is_visible_public: true,
    is_public_unlocked: false,
    progress_percent: 0,
    status: "planned",
    overview: "",
    overview_bn: "",
    key_features: "",
    key_features_bn: "",
    business_model: "",
    target_users: "",
    notes_internal: "",
  });

  const { data: module, isLoading } = useQuery({
    queryKey: ["hq-module-edit", id],
    queryFn: async () => {
      if (isNew) return null;
      const { data, error } = await supabase
        .from("seba_modules")
        .select("*")
        .eq("id", id)
        .single();
      
      if (error) throw error;
      return data;
    },
    enabled: !isNew,
  });

  useEffect(() => {
    if (module) {
      setFormData(module);
    }
  }, [module]);

  const saveMutation = useMutation({
    mutationFn: async (data: typeof formData) => {
      if (isNew) {
        const { error } = await supabase
          .from("seba_modules")
          .insert([data]);
        if (error) throw error;
      } else {
        const { error } = await supabase
          .from("seba_modules")
          .update(data)
          .eq("id", id);
        if (error) throw error;
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["hq-modules"] });
      queryClient.invalidateQueries({ queryKey: ["seba-modules"] });
      toast.success(isNew ? "Module created successfully" : "Module updated successfully");
      navigate("/hq");
    },
    onError: (error) => {
      toast.error("Failed to save module: " + error.message);
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async () => {
      const { error } = await supabase
        .from("seba_modules")
        .delete()
        .eq("id", id);
      if (error) throw error;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["hq-modules"] });
      queryClient.invalidateQueries({ queryKey: ["seba-modules"] });
      toast.success("Module deleted successfully");
      navigate("/hq");
    },
    onError: (error) => {
      toast.error("Failed to delete module: " + error.message);
    },
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    saveMutation.mutate(formData);
  };

  const handleDelete = () => {
    if (confirm("Are you sure you want to delete this module? This action cannot be undone.")) {
      deleteMutation.mutate();
    }
  };

  if (isLoading) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-background via-primary/5 to-secondary/5 flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-primary" />
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-background via-primary/5 to-secondary/5">
      <div className="container py-8">
        <div className="max-w-4xl mx-auto space-y-8">
          {/* Header */}
          <div className="flex items-center justify-between">
            <div>
              <Button variant="ghost" size="sm" asChild className="mb-2">
                <Link to="/hq">
                  <ArrowLeft className="mr-2 h-4 w-4" />
                  Back to Dashboard
                </Link>
              </Button>
              <h1 className="text-3xl font-bold">
                {isNew ? "Add New Module" : "Edit Module"}
              </h1>
            </div>
            {!isNew && (
              <Button variant="destructive" onClick={handleDelete} disabled={deleteMutation.isPending}>
                <Trash2 className="mr-2 h-4 w-4" />
                Delete
              </Button>
            )}
          </div>

          <form onSubmit={handleSubmit} className="space-y-6">
            {/* Basic Information */}
            <Card>
              <CardHeader>
                <CardTitle>Basic Information</CardTitle>
                <CardDescription>Core details about the module</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label htmlFor="slug">Slug (URL)</Label>
                    <Input
                      id="slug"
                      value={formData.slug}
                      onChange={(e) => setFormData({ ...formData, slug: e.target.value })}
                      placeholder="smart-module-name"
                      required
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="icon">Icon (Emoji)</Label>
                    <Input
                      id="icon"
                      value={formData.icon}
                      onChange={(e) => setFormData({ ...formData, icon: e.target.value })}
                      placeholder="📦"
                      maxLength={2}
                    />
                  </div>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="name">Name (English)</Label>
                  <Input
                    id="name"
                    value={formData.name}
                    onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                    required
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="name_bn">Name (Bengali)</Label>
                  <Input
                    id="name_bn"
                    value={formData.name_bn || ""}
                    onChange={(e) => setFormData({ ...formData, name_bn: e.target.value })}
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="short_tagline">Short Tagline (English)</Label>
                  <Input
                    id="short_tagline"
                    value={formData.short_tagline || ""}
                    onChange={(e) => setFormData({ ...formData, short_tagline: e.target.value })}
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="short_tagline_bn">Short Tagline (Bengali)</Label>
                  <Input
                    id="short_tagline_bn"
                    value={formData.short_tagline_bn || ""}
                    onChange={(e) => setFormData({ ...formData, short_tagline_bn: e.target.value })}
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="category">Category</Label>
                  <Input
                    id="category"
                    value={formData.category}
                    onChange={(e) => setFormData({ ...formData, category: e.target.value })}
                    placeholder="e.g., Core, Education, Health"
                    required
                  />
                </div>
              </CardContent>
            </Card>

            {/* Status & Progress */}
            <Card>
              <CardHeader>
                <CardTitle>Status & Progress</CardTitle>
                <CardDescription>Development status and progress tracking</CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="space-y-2">
                  <Label htmlFor="status">Status</Label>
                  <Select value={formData.status} onValueChange={(value) => setFormData({ ...formData, status: value })}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="planned">Planned</SelectItem>
                      <SelectItem value="in-progress">In Progress</SelectItem>
                      <SelectItem value="live">Live</SelectItem>
                      <SelectItem value="paused">Paused</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div className="space-y-2">
                  <div className="flex justify-between">
                    <Label>Progress</Label>
                    <span className="text-sm text-muted-foreground">{formData.progress_percent}%</span>
                  </div>
                  <Slider
                    value={[formData.progress_percent]}
                    onValueChange={([value]) => setFormData({ ...formData, progress_percent: value })}
                    max={100}
                    step={5}
                  />
                </div>

                <Separator />

                <div className="flex items-center justify-between">
                  <div className="space-y-0.5">
                    <Label>Visible to Public</Label>
                    <p className="text-sm text-muted-foreground">Show this module on public website</p>
                  </div>
                  <Switch
                    checked={formData.is_visible_public}
                    onCheckedChange={(checked) => setFormData({ ...formData, is_visible_public: checked })}
                  />
                </div>

                <div className="flex items-center justify-between">
                  <div className="space-y-0.5">
                    <Label>Publicly Unlocked</Label>
                    <p className="text-sm text-muted-foreground">Show full details to public users</p>
                  </div>
                  <Switch
                    checked={formData.is_public_unlocked}
                    onCheckedChange={(checked) => setFormData({ ...formData, is_public_unlocked: checked })}
                  />
                </div>
              </CardContent>
            </Card>

            {/* Detailed Content */}
            <Card>
              <CardHeader>
                <CardTitle>Detailed Content</CardTitle>
                <CardDescription>Full descriptions and information</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="overview">Overview (English)</Label>
                  <Textarea
                    id="overview"
                    value={formData.overview || ""}
                    onChange={(e) => setFormData({ ...formData, overview: e.target.value })}
                    rows={4}
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="overview_bn">Overview (Bengali)</Label>
                  <Textarea
                    id="overview_bn"
                    value={formData.overview_bn || ""}
                    onChange={(e) => setFormData({ ...formData, overview_bn: e.target.value })}
                    rows={4}
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="key_features">Key Features (one per line)</Label>
                  <Textarea
                    id="key_features"
                    value={formData.key_features || ""}
                    onChange={(e) => setFormData({ ...formData, key_features: e.target.value })}
                    rows={6}
                    placeholder="✓ Feature 1&#10;✓ Feature 2&#10;✓ Feature 3"
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="business_model">Business Model</Label>
                  <Textarea
                    id="business_model"
                    value={formData.business_model || ""}
                    onChange={(e) => setFormData({ ...formData, business_model: e.target.value })}
                    rows={3}
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="target_users">Target Users</Label>
                  <Textarea
                    id="target_users"
                    value={formData.target_users || ""}
                    onChange={(e) => setFormData({ ...formData, target_users: e.target.value })}
                    rows={2}
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="notes_internal">Internal Notes (HQ Only)</Label>
                  <Textarea
                    id="notes_internal"
                    value={formData.notes_internal || ""}
                    onChange={(e) => setFormData({ ...formData, notes_internal: e.target.value })}
                    rows={4}
                    placeholder="Private notes for HQ team only"
                  />
                </div>
              </CardContent>
            </Card>

            {/* Actions */}
            <Card>
              <CardContent className="pt-6">
                <div className="flex gap-4">
                  <Button type="submit" className="flex-1" disabled={saveMutation.isPending}>
                    <Save className="mr-2 h-4 w-4" />
                    {saveMutation.isPending ? "Saving..." : "Save Module"}
                  </Button>
                  <Button type="button" variant="outline" onClick={() => navigate("/hq")}>
                    Cancel
                  </Button>
                </div>
              </CardContent>
            </Card>
          </form>
        </div>
      </div>
    </div>
  );
}
