import { useNavigate } from "react-router-dom";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { SmartFriendHeader } from "@/components/SmartFriendHeader";
import CoreBrainWidget from "@/components/CoreBrainWidget";
import { QuickTools } from "@/components/home/QuickTools";
import { RecentActions } from "@/components/home/RecentActions";
import { RecommendedForYou } from "@/components/home/RecommendedForYou";
import { PinnedModules } from "@/components/home/PinnedModules";
import { FileCode, Sparkles, Package, Palette, MessageCircle, BookOpen, Briefcase, Settings as SettingsIcon } from "lucide-react";
import { cn } from "@/lib/utils";
import { Skeleton } from "@/components/ui/skeleton";
import { useOnboarding } from "@/hooks/useOnboarding";
import { useIdentityEngine } from "@/hooks/useIdentityEngine";
import { useEffect } from "react";

const Index = () => {
  const navigate = useNavigate();
  const { isFirstLaunch, onboarding } = useOnboarding();
  const { role, language } = useIdentityEngine();

  // Redirect to onboarding if first launch
  useEffect(() => {
    if (isFirstLaunch) {
      navigate('/onboarding');
    }
  }, [isFirstLaunch, navigate]);

  const isBangla = language === 'bn' || onboarding.language === 'bn';

  // Greeting based on role
  const getGreeting = () => {
    const timeOfDay = new Date().getHours();
    let timeGreeting = '';
    
    if (timeOfDay < 12) {
      timeGreeting = isBangla ? 'সুপ্রভাত' : 'Good Morning';
    } else if (timeOfDay < 18) {
      timeGreeting = isBangla ? 'শুভ অপরাহ্ণ' : 'Good Afternoon';
    } else {
      timeGreeting = isBangla ? 'শুভ সন্ধ্যা' : 'Good Evening';
    }

    if (role === 'owner') {
      return `${timeGreeting}, ${isBangla ? 'তাসনিম' : 'Tasnim'}!`;
    } else if (role === 'ava') {
      return `${timeGreeting}, ${onboarding.username || 'Ava'}!`;
    } else {
      return `${timeGreeting}${onboarding.username ? `, ${onboarding.username}` : ''}!`;
    }
  };

  const quickActions = [
    { 
      title: "Smart Chat", 
      titleBn: "স্মার্ট চ্যাট", 
      icon: MessageCircle, 
      route: "/chat", 
      desc: isBangla ? "AI সাথে কথা বলুন" : "Talk with AI",
      gradient: "from-blue-500 to-cyan-500"
    },
    { 
      title: "Study", 
      titleBn: "পড়াশোনা", 
      icon: BookOpen, 
      route: "/academy", 
      desc: isBangla ? "শিক্ষা সহায়তা" : "Learning support",
      gradient: "from-green-500 to-emerald-500"
    },
    { 
      title: "Modules", 
      titleBn: "মডিউল", 
      icon: Package, 
      route: "/modules", 
      desc: isBangla ? "সকল মডিউল" : "All modules",
      gradient: "from-purple-500 to-pink-500"
    },
    { 
      title: "Settings", 
      titleBn: "সেটিংস", 
      icon: SettingsIcon, 
      route: "/settings", 
      desc: isBangla ? "অ্যাপ সেটিংস" : "App settings",
      gradient: "from-orange-500 to-red-500"
    },
  ];

  const studioFeatures = [
    { 
      title: "Open IDE", 
      description: "Full-featured code editor", 
      icon: FileCode, 
      route: "/studio/ide", 
      gradient: "from-blue-500 to-cyan-500" 
    },
    { 
      title: "AI Engine", 
      description: "Generate code with AI", 
      icon: Sparkles, 
      route: "/studio/ai-engine", 
      gradient: "from-purple-500 to-pink-500" 
    },
    { 
      title: "Create Module", 
      description: "50+ ready modules", 
      icon: Package, 
      route: "/studio/modules", 
      gradient: "from-orange-500 to-red-500" 
    },
    { 
      title: "Website Designer", 
      description: "AI-powered design", 
      icon: Palette, 
      route: "/studio/designer", 
      gradient: "from-green-500 to-emerald-500" 
    },
  ];

  return (
    <div className="h-full overflow-auto pb-20">
      <div className="p-6 max-w-4xl mx-auto space-y-8">
        {/* Modern Premium Header */}
        <div>
          <SmartFriendHeader />
        </div>

        {/* Smart Friend ULTRA v2.0 Banner */}
        <Card className="bg-gradient-to-br from-primary/10 via-secondary/10 to-background border-primary/20 shadow-xl">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Sparkles className="h-5 w-5 text-primary" />
              Smart Friend ULTRA v2.0
            </CardTitle>
            <CardDescription>Your AI-powered development studio</CardDescription>
          </CardHeader>
          <CardContent>
            <Button onClick={() => navigate('/studio-v2')} className="w-full gap-2" size="lg">
              <FileCode className="h-5 w-5" />
              Open ULTRA Studio
            </Button>
          </CardContent>
        </Card>

        {/* Smart Seba HQ - Module Ecosystem */}
        <Card className="bg-gradient-to-br from-green-500/10 via-emerald-500/10 to-background border-green-500/20 shadow-xl">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Package className="h-5 w-5 text-green-500" />
              Smart Seba HQ
            </CardTitle>
            <CardDescription>50+ AI-Powered Smart Modules Ecosystem</CardDescription>
          </CardHeader>
          <CardContent className="space-y-3">
            <p className="text-sm text-muted-foreground">
              Explore our comprehensive ecosystem of AI-powered modules for education, health, business, commerce, and more. Built in Bangladesh for the world.
            </p>
            <Button onClick={() => navigate('/seba')} className="w-full gap-2" size="lg" variant="default">
              <Sparkles className="h-5 w-5" />
              Explore Smart Seba Modules
            </Button>
          </CardContent>
        </Card>

        {/* Quick Tools Section */}
        <QuickTools />

        {/* Recent Actions Section */}
        <RecentActions />

        {/* Recommended for You Section */}
        <RecommendedForYou />

        {/* Studio Quick Actions */}
        <div className="space-y-4">
          <h2 className="text-lg font-semibold">Development Tools</h2>
          <div className="grid grid-cols-2 gap-4">
            {studioFeatures.map((feature, index) => (
              <Card 
                key={index}
                className="cursor-pointer hover:shadow-xl transition-all hover:scale-105"
                onClick={() => navigate(feature.route)}
              >
                <CardHeader className="pb-3">
                  <div className={cn("w-10 h-10 rounded-lg bg-gradient-to-br flex items-center justify-center mb-2", feature.gradient)}>
                    <feature.icon className="h-5 w-5 text-white" />
                  </div>
                  <CardTitle className="text-sm">{feature.title}</CardTitle>
                  <CardDescription className="text-xs">{feature.description}</CardDescription>
                </CardHeader>
              </Card>
            ))}
          </div>
        </div>

        {/* Quick Actions */}
        <div className="space-y-4">
          <h2 className="text-lg font-semibold">Quick Actions</h2>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            {quickActions.map((action) => {
              const IconComponent = action.icon;
              return (
                <Card
                  key={action.route}
                  className="cursor-pointer hover:shadow-lg transition-shadow"
                  onClick={() => navigate(action.route)}
                >
                  <CardHeader>
                    <div className={cn("w-12 h-12 rounded-lg bg-gradient-to-br flex items-center justify-center mb-2", action.gradient)}>
                      <IconComponent className="h-6 w-6 text-white" />
                    </div>
                    <CardTitle className="text-lg">{action.titleBn}</CardTitle>
                    <CardDescription className="text-xs">{action.title}</CardDescription>
                  </CardHeader>
                  <CardContent>
                    <p className="text-sm text-muted-foreground">{action.desc}</p>
                  </CardContent>
                </Card>
              );
            })}
          </div>
        </div>

        {/* Core Brain System Monitor Widget */}
        <div>
          <CoreBrainWidget />
        </div>

        {/* Welcome Card */}
        <Card className="bg-primary/5 border-primary/20">
          <CardHeader>
            <CardTitle>স্বাগতম! | Welcome!</CardTitle>
          </CardHeader>
          <CardContent className="space-y-2">
            <p>✅ চ্যাট করুন বাংলা বা ইংরেজিতে</p>
            <p>✅ ১৩টি স্মার্ট মডিউল ব্যবহার করুন</p>
            <p>✅ আপনার প্রোফাইল কাস্টমাইজ করুন</p>
            <p>✅ রিমাইন্ডার সেট করুন (শীঘ্রই আসছে)</p>
            <p>✅ AI Code Generator সহ 50+ মডিউল</p>
            <Button className="w-full mt-4" onClick={() => navigate("/chat")}>
              শুরু করুন | Get Started
            </Button>
          </CardContent>
        </Card>
      </div>
    </div>
  );
};

export default Index;
