import { useParams, useNavigate } from "react-router-dom";
import { useQuery } from "@tanstack/react-query";
import { supabase } from "@/integrations/supabase/client";
import { useOwnerCheck } from "@/hooks/useOwnerCheck";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { ArrowLeft, Lock, Edit, CheckCircle2, Clock, Rocket } from "lucide-react";
import * as Icons from "lucide-react";
import { SebaFooter } from "@/components/seba/SebaFooter";

export default function ModuleDetail() {
  const { slug } = useParams();
  const navigate = useNavigate();
  const { isOwner } = useOwnerCheck();

  const { data: module, isLoading } = useQuery({
    queryKey: ["seba-module", slug],
    queryFn: async () => {
      const { data, error } = await supabase
        .from("seba_modules")
        .select("*")
        .eq("slug", slug)
        .single();

      if (error) throw error;
      return data;
    },
  });

  if (isLoading) {
    return (
      <div className="min-h-screen bg-background">
        <div className="container mx-auto px-4 py-8">
          <Skeleton className="h-10 w-32 mb-8" />
          <div className="grid lg:grid-cols-3 gap-8">
            <div className="lg:col-span-2 space-y-6">
              <Skeleton className="h-64" />
              <Skeleton className="h-96" />
            </div>
            <Skeleton className="h-96" />
          </div>
        </div>
      </div>
    );
  }

  if (!module) {
    return (
      <div className="min-h-screen bg-background flex items-center justify-center">
        <Card className="max-w-md w-full">
          <CardHeader>
            <CardTitle>Module Not Found</CardTitle>
            <CardDescription>
              The module you're looking for doesn't exist or has been removed.
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Button onClick={() => navigate("/seba")} className="w-full">
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to Home
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  const canViewFullDetails = isOwner || module.is_public_unlocked;

  // Get icon
  const getIcon = () => {
    if (module.icon && module.icon in Icons) {
      const IconComponent = Icons[module.icon as keyof typeof Icons] as any;
      return <IconComponent className="w-12 h-12" />;
    }
    return <Icons.Box className="w-12 h-12" />;
  };

  // Get status info
  const getStatusInfo = () => {
    if (module.is_public_unlocked) {
      return {
        icon: <CheckCircle2 className="w-5 h-5" />,
        label: "Available Now",
        variant: "default" as const,
        className: "bg-success text-success-foreground",
      };
    }
    if (module.status === "active" || module.status === "live") {
      return {
        icon: <Rocket className="w-5 h-5" />,
        label: "Live",
        variant: "default" as const,
        className: "bg-primary text-primary-foreground",
      };
    }
    return {
      icon: <Clock className="w-5 h-5" />,
      label: "Coming Soon",
      variant: "secondary" as const,
      className: "",
    };
  };

  const statusInfo = getStatusInfo();

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="border-b border-border/50 bg-card">
        <div className="container mx-auto px-4 py-6">
          <div className="flex items-center justify-between">
            <Button
              variant="ghost"
              onClick={() => navigate("/seba")}
              className="gap-2"
            >
              <ArrowLeft className="w-4 h-4" />
              Back to Modules
            </Button>

            {isOwner && (
              <Button
                variant="outline"
                onClick={() => navigate(`/hq-module-edit/${module.id}`)}
                className="gap-2"
              >
                <Edit className="w-4 h-4" />
                Edit Module
              </Button>
            )}
          </div>
        </div>
      </div>

      {/* Content */}
      <div className="container mx-auto px-4 py-12">
        <div className="grid lg:grid-cols-3 gap-8">
          {/* Main Content */}
          <div className="lg:col-span-2 space-y-6">
            {/* Module Header Card */}
            <Card className="border-2">
              <CardHeader className="space-y-6">
                <div className="flex items-start gap-6">
                  <div className="p-4 rounded-2xl bg-gradient-to-br from-primary/10 to-accent/10 text-primary">
                    {getIcon()}
                  </div>
                  <div className="flex-1 space-y-3">
                    <div className="flex items-center gap-3 flex-wrap">
                      <h1 className="text-3xl md:text-4xl font-bold">{module.name}</h1>
                      <Badge className={`gap-1 ${statusInfo.className}`}>
                        {statusInfo.icon}
                        {statusInfo.label}
                      </Badge>
                    </div>
                    {module.name_bn && (
                      <p className="text-xl text-muted-foreground font-medium">{module.name_bn}</p>
                    )}
                    <div className="flex items-center gap-2">
                      <Badge variant="outline">{module.category}</Badge>
                      {isOwner && module.progress_percent !== undefined && (
                        <Badge variant="secondary">{module.progress_percent}% Complete</Badge>
                      )}
                    </div>
                  </div>
                </div>

                {module.short_tagline && (
                  <p className="text-lg text-muted-foreground leading-relaxed">
                    {module.short_tagline}
                  </p>
                )}
              </CardHeader>
            </Card>

            {/* Locked Module View */}
            {!canViewFullDetails && (
              <Card className="border-2 border-dashed">
                <CardContent className="flex flex-col items-center justify-center py-16 text-center space-y-6">
                  <div className="p-6 rounded-full bg-secondary">
                    <Lock className="w-12 h-12 text-muted-foreground" />
                  </div>
                  <div className="space-y-2">
                    <h3 className="text-2xl font-bold">Module Under Development</h3>
                    <p className="text-muted-foreground max-w-md">
                      This module is currently being developed. Check back soon for more details and availability.
                    </p>
                  </div>
                  <Button size="lg" onClick={() => navigate("/seba")}>
                    Explore Other Modules
                  </Button>
                </CardContent>
              </Card>
            )}

            {/* Full Details (Owner or Public Unlocked) */}
            {canViewFullDetails && (
              <>
                {/* Overview */}
                {module.overview && (
                  <Card>
                    <CardHeader>
                      <CardTitle>Overview</CardTitle>
                    </CardHeader>
                    <CardContent className="prose prose-slate max-w-none dark:prose-invert">
                      <p className="text-muted-foreground leading-relaxed whitespace-pre-wrap">
                        {module.overview}
                      </p>
                    </CardContent>
                  </Card>
                )}

                {/* Key Features */}
                {module.key_features && (
                  <Card>
                    <CardHeader>
                      <CardTitle>Key Features</CardTitle>
                    </CardHeader>
                    <CardContent className="prose prose-slate max-w-none dark:prose-invert">
                      <div className="text-muted-foreground leading-relaxed whitespace-pre-wrap">
                        {module.key_features}
                      </div>
                    </CardContent>
                  </Card>
                )}

                {/* Business Model */}
                {module.business_model && (
                  <Card>
                    <CardHeader>
                      <CardTitle>Business Model</CardTitle>
                    </CardHeader>
                    <CardContent className="prose prose-slate max-w-none dark:prose-invert">
                      <p className="text-muted-foreground leading-relaxed whitespace-pre-wrap">
                        {module.business_model}
                      </p>
                    </CardContent>
                  </Card>
                )}

                {/* Internal Notes (Owner Only) */}
                {isOwner && module.notes_internal && (
                  <Card className="border-warning/50 bg-warning/5">
                    <CardHeader>
                      <CardTitle className="flex items-center gap-2">
                        <Lock className="w-5 h-5" />
                        Internal Notes (Owner Only)
                      </CardTitle>
                    </CardHeader>
                    <CardContent className="prose prose-slate max-w-none dark:prose-invert">
                      <p className="text-muted-foreground leading-relaxed whitespace-pre-wrap">
                        {module.notes_internal}
                      </p>
                    </CardContent>
                  </Card>
                )}
              </>
            )}
          </div>

          {/* Sidebar */}
          <div className="space-y-6">
            {/* Quick Info */}
            <Card>
              <CardHeader>
                <CardTitle>Module Info</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <div className="text-sm text-muted-foreground mb-1">Category</div>
                  <Badge variant="outline" className="text-base">{module.category}</Badge>
                </div>

                <div>
                  <div className="text-sm text-muted-foreground mb-1">Status</div>
                  <Badge className={statusInfo.className}>
                    {statusInfo.label}
                  </Badge>
                </div>

                {isOwner && module.progress_percent !== undefined && (
                  <div>
                    <div className="flex justify-between text-sm text-muted-foreground mb-2">
                      <span>Development Progress</span>
                      <span className="font-semibold">{module.progress_percent}%</span>
                    </div>
                    <div className="w-full h-2 bg-secondary rounded-full overflow-hidden">
                      <div
                        className="h-full bg-gradient-to-r from-primary to-accent transition-all"
                        style={{ width: `${module.progress_percent}%` }}
                      />
                    </div>
                  </div>
                )}

                {canViewFullDetails && module.target_users && (
                  <div>
                    <div className="text-sm text-muted-foreground mb-2">Target Users</div>
                    <p className="text-sm leading-relaxed">{module.target_users}</p>
                  </div>
                )}
              </CardContent>
            </Card>

            {/* CTA Card */}
            {canViewFullDetails && (
              <Card className="bg-gradient-to-br from-primary/10 to-accent/10 border-primary/20">
                <CardHeader>
                  <CardTitle>Get Started</CardTitle>
                  <CardDescription>
                    Ready to use this module?
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-3">
                  <Button className="w-full" size="lg">
                    Launch Module
                  </Button>
                  <Button variant="outline" className="w-full">
                    Learn More
                  </Button>
                </CardContent>
              </Card>
            )}
          </div>
        </div>
      </div>

      {/* Footer */}
      <SebaFooter />
    </div>
  );
}
