import { useEffect, useState } from "react";
import { useNavigate } from "react-router-dom";
import { supabase } from "@/integrations/supabase/client";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { ModuleFilters, ModuleCategory } from "@/components/modules/ModuleFilters";
import { toast } from "sonner";
import { Loader2, Lock, CheckCircle2, Clock } from "lucide-react";
import { addRecentAction } from "@/components/home/RecentActions";

type Module = {
  id: string;
  name: string;
  name_bn: string;
  description: string;
  description_bn: string;
  icon: string;
  route: string;
};

const Modules = () => {
  const navigate = useNavigate();
  const [modules, setModules] = useState<Module[]>([]);
  const [loading, setLoading] = useState(true);
  const [activeCategory, setActiveCategory] = useState<ModuleCategory>("all");

  useEffect(() => {
    fetchModules();
  }, []);

  const fetchModules = async () => {
    try {
      const { data, error } = await supabase
        .from("modules")
        .select("*")
        .eq("is_active", true)
        .order("order_index");

      if (error) throw error;
      setModules(data || []);
    } catch (error: any) {
      toast.error("মডিউল লোড করতে ব্যর্থ | Failed to load modules");
    } finally {
      setLoading(false);
    }
  };

  const handleModuleClick = (module: Module) => {
    addRecentAction({
      type: "module",
      title: module.name_bn,
      route: module.route,
    });
    
    if (module.route === '/core-brain') {
      window.location.href = '/core-brain';
    } else {
      toast.info(`${module.name_bn} শীঘ্রই আসছে | ${module.name} coming soon`);
    }
  };
  
  const getStatusBadge = (route: string) => {
    if (route === '/core-brain') {
      return { label: "Live", variant: "default" as const, icon: CheckCircle2 };
    }
    return { label: "Roadmap", variant: "secondary" as const, icon: Clock };
  };

  const filteredModules = activeCategory === "all" 
    ? modules 
    : modules.filter(m => m.route.includes(activeCategory));

  if (loading) {
    return (
      <div className="h-full overflow-auto">
        <div className="p-6 max-w-6xl mx-auto space-y-6">
          <Skeleton className="h-12 w-64" />
          <Skeleton className="h-8 w-full" />
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {[...Array(6)].map((_, i) => (
              <Skeleton key={i} className="h-48" />
            ))}
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="h-full overflow-auto pb-20">
      <div className="p-6 max-w-6xl mx-auto space-y-6">
        {/* Header */}
        <div>
          <h1 className="text-3xl font-bold mb-2">Smart Modules</h1>
          <p className="text-muted-foreground">
            সকল স্মার্ট মডিউল | All Smart Modules
          </p>
        </div>

        {/* Filters */}
        <ModuleFilters 
          activeCategory={activeCategory} 
          onCategoryChange={setActiveCategory} 
        />

        {/* Modules Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
          {filteredModules.map((module) => {
            const statusBadge = getStatusBadge(module.route);
            const StatusIcon = statusBadge.icon;
            return (
              <Card
                key={module.id}
                className="cursor-pointer hover:shadow-lg transition-all duration-200 hover:scale-105 border-muted"
                onClick={() => handleModuleClick(module)}
              >
                <CardHeader>
                  <div className="flex items-start justify-between mb-2">
                    <div className="text-4xl">{module.icon}</div>
                    <Badge variant={statusBadge.variant} className="gap-1">
                      <StatusIcon className="h-3 w-3" />
                      {statusBadge.label}
                    </Badge>
                  </div>
                  <CardTitle className="line-clamp-1">{module.name_bn}</CardTitle>
                  <CardDescription className="text-xs">{module.name}</CardDescription>
                </CardHeader>
                <CardContent>
                  <p className="text-sm text-muted-foreground line-clamp-2">
                    {module.description_bn || "AI-powered smart module"}
                  </p>
                </CardContent>
              </Card>
            );
          })}
          
          {/* Code Builder Module */}
          <Card
            className="cursor-pointer hover:shadow-lg transition-shadow bg-gradient-to-br from-primary/10 to-primary/5 border-primary/20"
            onClick={() => navigate('/code-builder')}
          >
            <CardHeader>
              <div className="text-4xl mb-2">🏗️</div>
              <CardTitle>কোড বিল্ডার</CardTitle>
              <CardDescription className="text-xs">Code Builder</CardDescription>
            </CardHeader>
            <CardContent>
              <p className="text-sm text-muted-foreground">
                স্মার্ট ফ্রেন্ডের জন্য কোড তৈরি করুন | Build code for Smart Friend
              </p>
            </CardContent>
          </Card>

          {/* Smart Friend Studio Module */}
          <Card
            className="cursor-pointer hover:shadow-lg transition-shadow bg-gradient-to-br from-purple-500/10 to-blue-500/5 border-purple-500/20"
            onClick={() => navigate('/studio')}
          >
            <CardHeader>
              <div className="text-4xl mb-2">🚀</div>
              <CardTitle>ডেভেলপার স্টুডিও</CardTitle>
              <CardDescription className="text-xs">Developer Studio 2.0</CardDescription>
            </CardHeader>
            <CardContent>
              <p className="text-sm text-muted-foreground">
                সম্পূর্ণ IDE, AI সহায়ক এবং আরও অনেক কিছু | Full IDE with AI Assistant & More
              </p>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
};

export default Modules;