import { useState } from "react";
import { OwnerGuard } from "@/components/OwnerGuard";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Switch } from "@/components/ui/switch";
import { moduleLoader, ModuleConfig } from "@/core/module_loader";
import { Package, Plus, Sparkles } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { Input } from "@/components/ui/input";

const ModulesManager = () => {
  const [modules, setModules] = useState<ModuleConfig[]>(moduleLoader.listModules());
  const [newModuleName, setNewModuleName] = useState("");
  const { toast } = useToast();

  const refreshModules = () => {
    setModules(moduleLoader.listModules());
  };

  const toggleModule = (slug: string, enabled: boolean) => {
    if (enabled) {
      moduleLoader.enableModule(slug);
    } else {
      moduleLoader.disableModule(slug);
    }
    refreshModules();
    toast({
      title: enabled ? "Module Enabled" : "Module Disabled",
      description: `Module ${slug} has been ${enabled ? "enabled" : "disabled"}`,
    });
  };

  const generateModule = () => {
    if (!newModuleName.trim()) {
      toast({
        title: "Module Name Required",
        description: "Please enter a module name",
        variant: "destructive",
      });
      return;
    }

    const newModule = moduleLoader.generateModule(newModuleName);
    refreshModules();
    setNewModuleName("");
    
    toast({
      title: "Module Generated",
      description: `Module "${newModule.name}" created successfully`,
    });
  };

  return (
    <OwnerGuard>
      <div className="min-h-screen bg-background p-6">
        <div className="max-w-7xl mx-auto space-y-6">
          <div className="flex items-center gap-3">
            <Package className="h-8 w-8 text-primary" />
            <div>
              <h1 className="text-3xl font-bold text-foreground">Modules Manager</h1>
              <p className="text-muted-foreground">Manage and generate system modules</p>
            </div>
          </div>

          {/* Generate New Module */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Sparkles className="h-5 w-5" />
                Generate New Module
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="flex gap-2">
                <Input
                  placeholder="Enter module template name..."
                  value={newModuleName}
                  onChange={(e) => setNewModuleName(e.target.value)}
                  onKeyDown={(e) => e.key === "Enter" && generateModule()}
                />
                <Button onClick={generateModule}>
                  <Plus className="h-4 w-4 mr-2" />
                  Generate
                </Button>
              </div>
              <p className="text-xs text-muted-foreground mt-2">
                AI will generate a module based on your template name
              </p>
            </CardContent>
          </Card>

          {/* Module List */}
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
            {modules.map((module) => (
              <Card key={module.slug}>
                <CardHeader>
                  <div className="flex items-start justify-between">
                    <div>
                      <CardTitle className="text-lg">{module.name}</CardTitle>
                      <p className="text-xs text-muted-foreground mt-1">
                        {module.description}
                      </p>
                    </div>
                    <Switch
                      checked={module.enabled}
                      onCheckedChange={(checked) => toggleModule(module.slug, checked)}
                    />
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="space-y-2">
                    <Badge variant={module.enabled ? "default" : "secondary"}>
                      {module.enabled ? "Enabled" : "Disabled"}
                    </Badge>
                    <div className="text-xs text-muted-foreground space-y-1">
                      <p>Slug: {module.slug}</p>
                      {module.version && <p>Version: {module.version}</p>}
                      {module.author && <p>Author: {module.author}</p>}
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </div>
    </OwnerGuard>
  );
};

export default ModulesManager;
