import { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Brain, User, Baby, Users, Globe, MapPin, Grid3x3, Check } from 'lucide-react';
import { useOnboarding } from '@/hooks/useOnboarding';
import { useIdentityEngine } from '@/hooks/useIdentityEngine';
import { SmartFriendIcon } from '@/components/SmartFriendIcon';

const MODULES = [
  { id: 'academy', name: 'Smart Academy', nameBn: 'স্মার্ট একাডেমি', icon: '📚' },
  { id: 'design', name: 'Design Hub', nameBn: 'ডিজাইন হাব', icon: '🎨' },
  { id: 'marketer', name: 'Smart Marketer', nameBn: 'স্মার্ট মার্কেটার', icon: '📊' },
  { id: 'doctor', name: 'Smart Doctor', nameBn: 'স্মার্ট ডাক্তার', icon: '🏥' },
  { id: 'news', name: 'Smart News', nameBn: 'স্মার্ট নিউজ', icon: '📰' },
  { id: 'bazar', name: 'Smart Bazar', nameBn: 'স্মার্ট বাজার', icon: '🛒' },
  { id: 'tourist', name: 'Smart Tourist', nameBn: 'স্মার্ট ট্যুরিস্ট', icon: '✈️' },
  { id: 'lawyer', name: 'Smart Lawyer', nameBn: 'স্মার্ট আইনজীবী', icon: '⚖️' },
  { id: 'school', name: 'Smart School', nameBn: 'স্মার্ট স্কুল', icon: '🏫' },
  { id: 'call', name: 'Call Center', nameBn: 'কল সেন্টার', icon: '📞' },
  { id: 'ad', name: 'AI Ad Firm', nameBn: 'এআই বিজ্ঞাপন', icon: '📺' },
  { id: 'job', name: 'Smart Job', nameBn: 'স্মার্ট চাকরি', icon: '💼' },
];

const Onboarding = () => {
  const navigate = useNavigate();
  const { completeOnboarding } = useOnboarding();
  const { switchRole, setLanguage: setIdentityLanguage } = useIdentityEngine();
  
  const [step, setStep] = useState(1);
  const [role, setRole] = useState<'owner' | 'ava' | 'guest'>('owner');
  const [username, setUsername] = useState('');
  const [language, setLanguage] = useState('bn');
  const [region, setRegion] = useState('BD');
  const [pinnedModules, setPinnedModules] = useState<string[]>([]);

  const handleModuleToggle = (moduleId: string) => {
    if (pinnedModules.includes(moduleId)) {
      setPinnedModules(pinnedModules.filter(id => id !== moduleId));
    } else if (pinnedModules.length < 6) {
      setPinnedModules([...pinnedModules, moduleId]);
    }
  };

  const handleComplete = () => {
    completeOnboarding({
      role,
      language,
      region,
      pinnedModules,
      username: username || undefined,
    });

    switchRole(role === 'guest' ? 'owner' : role);
    setIdentityLanguage(language);

    navigate('/');
  };

  const isBangla = language === 'bn';

  return (
    <div className="min-h-screen bg-gradient-to-br from-primary/10 via-background to-secondary/10 flex items-center justify-center p-4">
      <Card className="w-full max-w-2xl">
        {step === 1 && (
          <>
            <CardHeader className="text-center space-y-4">
              <div className="flex justify-center">
                <div className="p-4 bg-gradient-to-br from-primary/20 to-secondary/20 rounded-full">
                  <Brain className="h-16 w-16 text-primary" />
                </div>
              </div>
              <CardTitle className="text-3xl">
                {isBangla ? 'স্বাগতম!' : 'Welcome!'}
              </CardTitle>
              <CardDescription className="text-lg">
                {isBangla 
                  ? 'আপনার ব্যক্তিগত বুদ্ধিমান সহায়ক — Smart Friend Ultra.'
                  : 'Your personal intelligent assistant — Smart Friend Ultra.'}
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="space-y-4 text-center text-muted-foreground">
                <p>{isBangla ? '✨ অফলাইন ও অনলাইন একসাথে কাজ করে' : '✨ Works both offline and online'}</p>
                <p>{isBangla ? '🛡️ আপনার পরিচয় ও নিরাপত্তা সুরক্ষিত' : '🛡️ Your identity and privacy protected'}</p>
                <p>{isBangla ? '🌍 বাংলা ও ইংরেজি উভয় ভাষায় সাহায্য' : '🌍 Support in Bangla and English'}</p>
              </div>
              <Button onClick={() => setStep(2)} className="w-full" size="lg">
                {isBangla ? 'শুরু করুন' : 'Get Started'}
              </Button>
            </CardContent>
          </>
        )}

        {step === 2 && (
          <>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Users className="h-6 w-6" />
                {isBangla ? 'আপনি কে?' : 'Who are you?'}
              </CardTitle>
              <CardDescription>{isBangla ? 'আপনার ভূমিকা নির্বাচন করুন' : 'Select your role'}</CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <RadioGroup value={role} onValueChange={(v) => setRole(v as any)}>
                <div className="flex items-center space-x-3 p-4 border rounded-lg hover:bg-muted/50 cursor-pointer">
                  <RadioGroupItem value="owner" id="owner" />
                  <Label htmlFor="owner" className="flex-1 cursor-pointer">
                    <div className="flex items-center gap-3">
                      <User className="h-5 w-5 text-primary" />
                      <div>
                        <div className="font-medium">{isBangla ? 'আমি Owner (তাসনিম)' : 'I am the Owner (Tasnim)'}</div>
                        <div className="text-sm text-muted-foreground">{isBangla ? 'পূর্ণ অ্যাক্সেস ও নিয়ন্ত্রণ' : 'Full access and control'}</div>
                      </div>
                    </div>
                  </Label>
                </div>

                <div className="flex items-center space-x-3 p-4 border rounded-lg hover:bg-muted/50 cursor-pointer">
                  <RadioGroupItem value="ava" id="ava" />
                  <Label htmlFor="ava" className="flex-1 cursor-pointer">
                    <div className="flex items-center gap-3">
                      <Baby className="h-5 w-5 text-secondary" />
                      <div>
                        <div className="font-medium">{isBangla ? 'আমি Ava' : 'I am Ava'}</div>
                        <div className="text-sm text-muted-foreground">{isBangla ? 'শিশু-নিরাপদ মোড' : 'Child-safe mode'}</div>
                      </div>
                    </div>
                  </Label>
                </div>

                <div className="flex items-center space-x-3 p-4 border rounded-lg hover:bg-muted/50 cursor-pointer">
                  <RadioGroupItem value="guest" id="guest" />
                  <Label htmlFor="guest" className="flex-1 cursor-pointer">
                    <div className="flex items-center gap-3">
                      <Users className="h-5 w-5 text-muted-foreground" />
                      <div>
                        <div className="font-medium">{isBangla ? 'অন্য ব্যবহারকারী (Guest)' : 'Another user (Guest)'}</div>
                        <div className="text-sm text-muted-foreground">{isBangla ? 'সাধারণ অ্যাক্সেস' : 'General access'}</div>
                      </div>
                    </div>
                  </Label>
                </div>
              </RadioGroup>

              {(role === 'guest' || role === 'ava') && (
                <div className="space-y-2">
                  <Label htmlFor="username">{isBangla ? 'আপনার নাম (ঐচ্ছিক)' : 'Your Name (Optional)'}</Label>
                  <Input
                    id="username"
                    value={username}
                    onChange={(e) => setUsername(e.target.value)}
                    placeholder={isBangla ? 'নাম লিখুন' : 'Enter name'}
                  />
                </div>
              )}

              <div className="flex gap-2">
                <Button onClick={() => setStep(1)} variant="outline" className="flex-1">{isBangla ? 'পূর্ববর্তী' : 'Back'}</Button>
                <Button onClick={() => setStep(3)} className="flex-1">{isBangla ? 'পরবর্তী' : 'Next'}</Button>
              </div>
            </CardContent>
          </>
        )}

        {step === 3 && (
          <>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Globe className="h-6 w-6" />
                {isBangla ? 'ভাষা ও অঞ্চল' : 'Language & Region'}
              </CardTitle>
              <CardDescription>{isBangla ? 'আপনার পছন্দের ভাষা এবং অঞ্চল নির্বাচন করুন' : 'Select your preferred language and region'}</CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="space-y-2">
                <Label htmlFor="language">{isBangla ? 'ভাষা' : 'Language'}</Label>
                <Select value={language} onValueChange={setLanguage}>
                  <SelectTrigger id="language"><SelectValue /></SelectTrigger>
                  <SelectContent>
                    <SelectItem value="bn">বাংলা (Bangla)</SelectItem>
                    <SelectItem value="en">English</SelectItem>
                    <SelectItem value="hi">हिन्दी (Hindi)</SelectItem>
                    <SelectItem value="ar">العربية (Arabic)</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label htmlFor="region">{isBangla ? 'অঞ্চল' : 'Region'}</Label>
                <Select value={region} onValueChange={setRegion}>
                  <SelectTrigger id="region"><SelectValue /></SelectTrigger>
                  <SelectContent>
                    <SelectItem value="BD">🇧🇩 Bangladesh</SelectItem>
                    <SelectItem value="IN">🇮🇳 India</SelectItem>
                    <SelectItem value="PK">🇵🇰 Pakistan</SelectItem>
                    <SelectItem value="AE">🇦🇪 UAE</SelectItem>
                    <SelectItem value="SA">🇸🇦 Saudi Arabia</SelectItem>
                    <SelectItem value="MY">🇲🇾 Malaysia</SelectItem>
                    <SelectItem value="US">🇺🇸 United States</SelectItem>
                    <SelectItem value="GB">🇬🇧 United Kingdom</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="flex gap-2">
                <Button onClick={() => setStep(2)} variant="outline" className="flex-1">{isBangla ? 'পূর্ববর্তী' : 'Back'}</Button>
                <Button onClick={() => setStep(4)} className="flex-1">{isBangla ? 'পরবর্তী' : 'Next'}</Button>
              </div>
            </CardContent>
          </>
        )}

        {step === 4 && (
          <>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Grid3x3 className="h-6 w-6" />
                {isBangla ? 'মডিউল পছন্দ' : 'Module Preferences'}
              </CardTitle>
              <CardDescription>{isBangla ? 'আপনার প্রিয় ৪-৬টি মডিউল নির্বাচন করুন' : 'Select 4-6 favorite modules for quick access'}</CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="grid grid-cols-2 gap-3">
                {MODULES.map((module) => {
                  const isSelected = pinnedModules.includes(module.id);
                  return (
                    <button
                      key={module.id}
                      onClick={() => handleModuleToggle(module.id)}
                      className={`p-4 border-2 rounded-lg transition-all ${
                        isSelected ? 'border-primary bg-primary/10' : 'border-border hover:border-primary/50'
                      } ${pinnedModules.length >= 6 && !isSelected ? 'opacity-50 cursor-not-allowed' : ''}`}
                      disabled={pinnedModules.length >= 6 && !isSelected}
                    >
                      <div className="text-3xl mb-2">{module.icon}</div>
                      <div className="text-sm font-medium">{isBangla ? module.nameBn : module.name}</div>
                      {isSelected && <Check className="h-4 w-4 text-primary mt-1 mx-auto" />}
                    </button>
                  );
                })}
              </div>

              <div className="text-center text-sm text-muted-foreground">
                {pinnedModules.length} / 6 {isBangla ? 'নির্বাচিত' : 'selected'}
              </div>

              <div className="flex gap-2">
                <Button onClick={() => setStep(3)} variant="outline" className="flex-1">{isBangla ? 'পূর্ববর্তী' : 'Back'}</Button>
                <Button onClick={() => setStep(5)} className="flex-1" disabled={pinnedModules.length < 4}>{isBangla ? 'পরবর্তী' : 'Next'}</Button>
              </div>
            </CardContent>
          </>
        )}

        {step === 5 && (
          <>
            <CardHeader className="text-center">
              <CardTitle className="flex items-center justify-center gap-2">
                <Check className="h-6 w-6 text-primary" />
                {isBangla ? 'সব প্রস্তুত!' : 'All Set!'}
              </CardTitle>
              <CardDescription>{isBangla ? 'আপনার সেটিংস পর্যালোচনা করুন' : 'Review your settings'}</CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="space-y-4">
                <div className="p-3 bg-muted rounded-lg">
                  <div className="text-sm font-medium mb-1">{isBangla ? 'ভূমিকা' : 'Role'}</div>
                  <div className="text-sm text-muted-foreground">
                    {role === 'owner' && (isBangla ? 'Owner (তাসনিম)' : 'Owner (Tasnim)')}
                    {role === 'ava' && 'Ava'}
                    {role === 'guest' && (username || (isBangla ? 'অতিথি' : 'Guest'))}
                  </div>
                </div>

                <div className="p-3 bg-muted rounded-lg">
                  <div className="text-sm font-medium mb-1">{isBangla ? 'ভাষা ও অঞ্চল' : 'Language & Region'}</div>
                  <div className="text-sm text-muted-foreground">
                    {language === 'bn' && 'বাংলা'} 
                    {language === 'en' && 'English'}
                    {language === 'hi' && 'हिन्दी'}
                    {language === 'ar' && 'العربية'}
                    {' • '}
                    {region}
                  </div>
                </div>

                <div className="p-3 bg-muted rounded-lg">
                  <div className="text-sm font-medium mb-2">{isBangla ? 'পিন করা মডিউল' : 'Pinned Modules'}</div>
                  <div className="flex flex-wrap gap-2">
                    {pinnedModules.map(id => {
                      const module = MODULES.find(m => m.id === id);
                      return module ? (
                        <span key={id} className="text-xs px-2 py-1 bg-primary/10 rounded">
                          {module.icon} {isBangla ? module.nameBn : module.name}
                        </span>
                      ) : null;
                    })}
                  </div>
                </div>
              </div>

              <div className="flex gap-2">
                <Button onClick={() => setStep(4)} variant="outline" className="flex-1">{isBangla ? 'পূর্ববর্তী' : 'Back'}</Button>
                <Button onClick={handleComplete} className="flex-1">{isBangla ? 'Smart Friend Ultra-তে প্রবেশ করুন' : 'Enter Smart Friend Ultra'}</Button>
              </div>
            </CardContent>
          </>
        )}

        <div className="px-6 pb-6">
          <div className="flex justify-center gap-2">
            {[1, 2, 3, 4, 5].map((s) => (
              <div
                key={s}
                className={`h-2 w-12 rounded-full transition-colors ${
                  s === step ? 'bg-primary' : s < step ? 'bg-primary/50' : 'bg-muted'
                }`}
              />
            ))}
          </div>
        </div>
      </Card>
    </div>
  );
};

export default Onboarding;
