import { useEffect, useState } from "react";
import { useAuth } from "@/hooks/useAuth";
import { supabase } from "@/integrations/supabase/client";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { toast } from "sonner";
import { Loader2, LogOut, Languages } from "lucide-react";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import { LanguageSelector } from "@/components/LanguageSelector";
import { getLanguage } from "@/lib/i18n/languages";

const Profile = () => {
  const { user, signOut } = useAuth();
  const [loading, setLoading] = useState(false);
  const [fullName, setFullName] = useState("");
  const [phone, setPhone] = useState("");
  const [language, setLanguage] = useState("bn");

  useEffect(() => {
    if (user) {
      fetchProfile();
    }
  }, [user]);

  const fetchProfile = async () => {
    try {
      const { data, error } = await supabase
        .from("profiles")
        .select("*")
        .eq("id", user?.id)
        .single();

      if (error) throw error;

      if (data) {
        setFullName(data.full_name || "");
        setPhone(data.phone || "");
        setLanguage(data.language_preference || "bn");
      }
    } catch (error) {
      console.error("Error fetching profile:", error);
    }
  };

  const handleUpdate = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      const { error } = await supabase
        .from("profiles")
        .update({
          full_name: fullName,
          phone: phone,
          language_preference: language,
        })
        .eq("id", user?.id);

      if (error) throw error;

      toast.success("প্রোফাইল আপডেট সফল | Profile updated successfully");
    } catch (error: any) {
      toast.error(error.message || "আপডেট ব্যর্থ | Update failed");
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="h-full overflow-auto">
      <div className="p-6 max-w-2xl mx-auto">
        <Card>
          <CardHeader>
            <div className="flex items-center justify-between">
              <div>
                <CardTitle>প্রোফাইল | Profile</CardTitle>
                <CardDescription>আপনার প্রোফাইল পরিচালনা করুন</CardDescription>
              </div>
              <Button variant="outline" size="sm" onClick={signOut}>
                <LogOut className="h-4 w-4 mr-2" />
                লগআউট | Logout
              </Button>
            </div>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleUpdate} className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="email">ইমেইল | Email</Label>
                <Input
                  id="email"
                  type="email"
                  value={user?.email || ""}
                  disabled
                  className="bg-muted"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="fullName">পূর্ণ নাম | Full Name</Label>
                <Input
                  id="fullName"
                  type="text"
                  value={fullName}
                  onChange={(e) => setFullName(e.target.value)}
                  placeholder="আপনার নাম | Your name"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="phone">ফোন | Phone</Label>
                <Input
                  id="phone"
                  type="tel"
                  value={phone}
                  onChange={(e) => setPhone(e.target.value)}
                  placeholder="+880..."
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="language">ভাষা পছন্দ | Language Preference</Label>
                <p className="text-sm text-muted-foreground mb-2">
                  Choose from 100+ languages. RTL support included.
                </p>
                <Dialog>
                  <DialogTrigger asChild>
                    <Button
                      type="button"
                      variant="outline"
                      className="w-full justify-start"
                    >
                      <Languages className="h-4 w-4 mr-2" />
                      {getLanguage(language)?.nativeName || language.toUpperCase()} ({getLanguage(language)?.name || language})
                    </Button>
                  </DialogTrigger>
                  <DialogContent className="max-w-4xl max-h-[90vh] overflow-hidden">
                    <DialogHeader>
                      <DialogTitle>Select Your Language</DialogTitle>
                      <DialogDescription>
                        Choose your preferred language from 100+ supported languages. 
                        RTL layouts are automatically enabled for Arabic, Hebrew, Urdu, etc.
                      </DialogDescription>
                    </DialogHeader>
                    <LanguageSelector
                      selectedLanguage={language}
                      onLanguageSelect={setLanguage}
                    />
                  </DialogContent>
                </Dialog>
              </div>

              <Button type="submit" className="w-full" disabled={loading}>
                {loading && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
                আপডেট করুন | Update Profile
              </Button>
            </form>
          </CardContent>
        </Card>
      </div>
    </div>
  );
};

export default Profile;