import { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { toast } from "sonner";
import { useSemanticCache } from "@/hooks/useSemanticCache";
import { useTheme } from "next-themes";
import { useLanguage } from "@/hooks/useLanguage";
import { ArrowLeft, Download, Trash2, Wifi, WifiOff, Moon, Sun, Shield, TrendingDown } from "lucide-react";
import { LanguageSelector } from "@/components/LanguageSelector";

const Settings = () => {
  const navigate = useNavigate();
  const { theme, setTheme } = useTheme();
  const { language, setLanguage } = useLanguage();
  const { getCacheStats, clearCache } = useSemanticCache();
  
  const [offlineMode, setOfflineMode] = useState(false);
  const [childSafeMode, setChildSafeMode] = useState(false);
  const [cacheStats, setCacheStats] = useState({ hits: 0, total: 0, hitRate: '0', savedCosts: '0' });

  useEffect(() => {
    // Load settings from localStorage
    const savedSettings = localStorage.getItem('smart_friend_settings');
    if (savedSettings) {
      const settings = JSON.parse(savedSettings);
      setOfflineMode(settings.offlineMode || false);
      setChildSafeMode(settings.childSafeMode || false);
    }
    
    // Load cache stats
    setCacheStats(getCacheStats());
  }, []);

  const saveSettings = () => {
    const settings = {
      offlineMode,
      childSafeMode,
      theme,
      language
    };
    localStorage.setItem('smart_friend_settings', JSON.stringify(settings));
    toast.success(language === 'bn' ? 'সেটিংস সংরক্ষিত হয়েছে' : 'Settings saved');
  };

  const exportSettings = () => {
    const settings = {
      offlineMode,
      childSafeMode,
      theme,
      language,
      exportDate: new Date().toISOString()
    };
    
    const dataStr = JSON.stringify(settings, null, 2);
    const dataBlob = new Blob([dataStr], { type: 'application/json' });
    const url = URL.createObjectURL(dataBlob);
    const link = document.createElement('a');
    link.href = url;
    link.download = 'smart-friend-settings.json';
    link.click();
    
    toast.success(language === 'bn' ? 'সেটিংস এক্সপোর্ট হয়েছে' : 'Settings exported');
  };

  const handleClearCache = () => {
    clearCache();
    setCacheStats({ hits: 0, total: 0, hitRate: '0', savedCosts: '0' });
    toast.success(language === 'bn' ? 'ক্যাশ পরিষ্কার হয়েছে' : 'Cache cleared');
  };

  return (
    <div className="h-full overflow-auto pb-20">
      <div className="p-6 max-w-2xl mx-auto space-y-6">
        {/* Header */}
        <div className="flex items-center gap-3">
          <Button variant="ghost" size="icon" onClick={() => navigate(-1)}>
            <ArrowLeft className="h-5 w-5" />
          </Button>
          <div>
            <h1 className="text-2xl font-bold">
              {language === 'bn' ? 'সেটিংস' : 'Settings'}
            </h1>
            <p className="text-sm text-muted-foreground">
              {language === 'bn' ? 'আপনার পছন্দ কনফিগার করুন' : 'Configure your preferences'}
            </p>
          </div>
        </div>

        {/* General Settings */}
        <Card>
          <CardHeader>
            <CardTitle>{language === 'bn' ? 'সাধারণ সেটিংস' : 'General Settings'}</CardTitle>
            <CardDescription>
              {language === 'bn' ? 'মৌলিক অ্যাপ্লিকেশন পছন্দ' : 'Basic application preferences'}
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            {/* Language */}
            <div className="space-y-2">
              <Label>{language === 'bn' ? 'ভাষা' : 'Language'}</Label>
              <LanguageSelector 
                selectedLanguage={language} 
                onLanguageSelect={setLanguage} 
              />
            </div>

            {/* Theme */}
            <div className="flex items-center justify-between">
              <div className="space-y-0.5">
                <Label>{language === 'bn' ? 'থিম' : 'Theme'}</Label>
                <p className="text-sm text-muted-foreground">
                  {language === 'bn' ? 'লাইট বা ডার্ক মোড' : 'Light or dark mode'}
                </p>
              </div>
              <div className="flex gap-2">
                <Button
                  variant={theme === 'light' ? 'default' : 'outline'}
                  size="sm"
                  onClick={() => setTheme('light')}
                >
                  <Sun className="h-4 w-4" />
                </Button>
                <Button
                  variant={theme === 'dark' ? 'default' : 'outline'}
                  size="sm"
                  onClick={() => setTheme('dark')}
                >
                  <Moon className="h-4 w-4" />
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Connectivity Settings */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              {offlineMode ? <WifiOff className="h-5 w-5" /> : <Wifi className="h-5 w-5" />}
              {language === 'bn' ? 'সংযোগ সেটিংস' : 'Connectivity Settings'}
            </CardTitle>
            <CardDescription>
              {language === 'bn' ? 'অনলাইন এবং অফলাইন পছন্দ' : 'Online and offline preferences'}
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex items-center justify-between">
              <div className="space-y-0.5">
                <Label>{language === 'bn' ? 'অফলাইন-ফার্স্ট মোড' : 'Offline-First Mode'}</Label>
                <p className="text-sm text-muted-foreground">
                  {language === 'bn' 
                    ? 'সম্ভব হলে ক্যাশড রেসপন্স ব্যবহার করুন' 
                    : 'Use cached responses when possible'}
                </p>
              </div>
              <Switch checked={offlineMode} onCheckedChange={setOfflineMode} />
            </div>
          </CardContent>
        </Card>

        {/* Safety Settings */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Shield className="h-5 w-5" />
              {language === 'bn' ? 'নিরাপত্তা সেটিংস' : 'Safety Settings'}
            </CardTitle>
            <CardDescription>
              {language === 'bn' ? 'শিশু-নিরাপদ এবং কন্টেন্ট ফিল্টার' : 'Child-safe and content filters'}
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex items-center justify-between">
              <div className="space-y-0.5">
                <Label>{language === 'bn' ? 'শিশু-নিরাপদ মোড' : 'Child-Safe Mode'}</Label>
                <p className="text-sm text-muted-foreground">
                  {language === 'bn' 
                    ? 'AVA-এর জন্য অতিরিক্ত কন্টেন্ট ফিল্টারিং' 
                    : 'Additional content filtering for AVA'}
                </p>
              </div>
              <Switch checked={childSafeMode} onCheckedChange={setChildSafeMode} />
            </div>
          </CardContent>
        </Card>

        {/* Cost Optimization Stats */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <TrendingDown className="h-5 w-5 text-green-500" />
              {language === 'bn' ? 'খরচ অপ্টিমাইজেশন' : 'Cost Optimization'}
            </CardTitle>
            <CardDescription>
              {language === 'bn' ? 'API ব্যবহার এবং সাশ্রয়' : 'API usage and savings'}
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-1">
                <p className="text-sm text-muted-foreground">
                  {language === 'bn' ? 'ক্যাশ হিট' : 'Cache Hits'}
                </p>
                <p className="text-2xl font-bold text-green-500">{cacheStats.hits}</p>
              </div>
              <div className="space-y-1">
                <p className="text-sm text-muted-foreground">
                  {language === 'bn' ? 'মোট কোয়েরি' : 'Total Queries'}
                </p>
                <p className="text-2xl font-bold">{cacheStats.total}</p>
              </div>
              <div className="space-y-1">
                <p className="text-sm text-muted-foreground">
                  {language === 'bn' ? 'হিট রেট' : 'Hit Rate'}
                </p>
                <p className="text-2xl font-bold">{cacheStats.hitRate}%</p>
              </div>
              <div className="space-y-1">
                <p className="text-sm text-muted-foreground">
                  {language === 'bn' ? 'আনুমানিক সাশ্রয়' : 'Est. Savings'}
                </p>
                <p className="text-2xl font-bold text-green-500">${cacheStats.savedCosts}</p>
              </div>
            </div>
            <Button variant="outline" size="sm" onClick={handleClearCache} className="w-full">
              <Trash2 className="h-4 w-4 mr-2" />
              {language === 'bn' ? 'ক্যাশ পরিষ্কার করুন' : 'Clear Cache'}
            </Button>
          </CardContent>
        </Card>

        {/* Actions */}
        <div className="flex gap-3">
          <Button onClick={saveSettings} className="flex-1">
            {language === 'bn' ? 'সেটিংস সংরক্ষণ করুন' : 'Save Settings'}
          </Button>
          <Button variant="outline" onClick={exportSettings}>
            <Download className="h-4 w-4 mr-2" />
            {language === 'bn' ? 'এক্সপোর্ট' : 'Export'}
          </Button>
        </div>
      </div>
    </div>
  );
};

export default Settings;
